unit Main;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	StdCtrls, ShellAPI,	Registry, ExtCtrls, ComCtrls, ImgList, FileCtrl, Menus,
	Plus4, Stringz;

Const
	KeyboardMatrix : Array [ 0 .. 63 ] Of String =
	(
		'ins/del', '3', '5', '7', '9', 'Down', 'Right', '1',
		'Return', 'w', 'r', 'y', 'i', 'p', '*', 'Clr/Home',
		'Pound', 'a', 'd', 'g', 'j', 'l', ']', 'Control',
		'Help/F7', '4', '6', '8', '0', 'Up', 'Left', '2',
		'F1/F4', 'z', 'c', 'b', 'm', '>', 'Escape', 'Space',
		'F2/F5', 's', 'f', 'h', 'k', '[', '=', 'Commodore',
		'F3/F6', 'e', 't', 'u', 'o', '-', '+', 'q',
		'@', 'Shift', 'x', 'v', 'n', '<', '/', 'Run/Stop'
	);

	ColorCodes : Array [ 0 .. 15 ] Of String =
	(
		'Black',
		'White',
		'Red',
		'Cyan',
		'Purple',
		'Green',
		'Blue',
		'Yellow',
		'Orange',
		'Brown',
		'Yellow-Green',
		'Pink',
		'Blue-Green',
		'Light Blue',
		'Dark Blue',
		'Light Green'
	);

	Max = $10000; // maximum memory size, 64KB

Type
	TMemory = Array [ 0 .. Max - 1 ] Of Byte;
	PMemory = ^TMemory;

	TFlags = Array [ 0 .. Max - 1 ] Of Boolean;
	PFlags = ^TFlags;

	TDataFormat = (
		dfAuto,
		dfByte,
		dfWord );

	TDataStyle = Record
		DataFormat : TDataFormat;
		Columns : Integer; // default: 16
		StopByte : Integer; // -1 = none
	End;

	TDataStyles = Array [ 0 .. Max - 1 ] Of TDataStyle;
	PDataStyles = ^TDataStyles;

	TMemCodes = ( mcNone, mcInstruction, mcParams );
	TMemCode = Array [ 0 .. Max - 1 ] Of TMemCodes;
	PMemCode = ^TMemCode;

	TLabelType = (
		ltNone,
		ltStart,
		ltJump,
		ltSub,
		ltBranch,
		ltIRQ,
		ltRes,
		ltNMI,
		ltVec,
		ltCharset );

	TLabels = Array [ 0 .. Max - 1 ] Of TLabelType;
	PLabels = ^TLabels;

	TSubParam = Record
		SpecialType : ( ssNone, ssLength, ssStopByte );
		Value : Byte;
	End;
	TSubParams = Array [ 0 .. Max - 1 ] Of TSubParam;
	PSubParams = ^TSubParams;

	TJumpTable = Record
		Address : Integer;
		Length : Integer;
		Style : Integer; // 0, 1, 2, 3 are valid
	End;

	TCommentByte = Record
		Position : Integer;
		MatchValue : String;
	End;

	TComment = Record
		Comment : String;
		nCommentBytes : Integer;
		CommentBytes : Array [ 1 .. 16 ] Of TCommentByte;
	End;

	TFormMain = class(TForm)

		DisAssMenu: TMainMenu;

		mnuFile: TMenuItem;
		mnuFileOpen: TMenuItem;
		mnuFileDiv1: TMenuItem;
		mnuFileExit: TMenuItem;
		mnuHelp: TMenuItem;
		mnuHelpContents: TMenuItem;
		mnuHelpDiv1: TMenuItem;
		mnuHelpAbout: TMenuItem;

		OpenDialog1: TOpenDialog;
		BevelTopDivider: TBevel;
		PageControlMain: TPageControl;

		TabImages: TImageList;
		TabSheetSettings: TTabSheet;
		TabSheetStatistics: TTabSheet;
    TabSheetPreview: TTabSheet;

		Animate1: TAnimate;
    PanelSizeLimit: TPanel;
		LabelSizeLimit: TLabel;
		optSizeLimit1: TRadioButton;
		optSizeLimit2: TRadioButton;
		EditSizeLimit: TEdit;
		UpDownSizeLimit: TUpDown;
		LabelSizeLimitKB: TLabel;
		LabelTargetExtension: TLabel;
		EditTargetExt: TEdit;
		EditDestFolder: TEdit;
		LabelDestFolder: TLabel;

		MemoPreview: TMemo;
		LabelProgramName: TLabel;
		EditProgramName: TEdit;
		LabelMemoryArea: TLabel;
		EditMemBeg: TEdit;
		EditMemEnd: TEdit;
		LabelStartAddress: TLabel;
		EditMemStart: TEdit;
		MemoStats: TMemo;
		StatusBarMain: TStatusBar;
    LabelSource: TLabel;
    PanelTargetAssembler: TPanel;
		optTarget1: TRadioButton;
		optTarget2: TRadioButton;
		optTarget3: TRadioButton;
    LabelTargetAssembler: TLabel;
		cmdBrowse: TButton;
		cmdStart: TButton;
		mnuActions: TMenuItem;
		mnuReg: TMenuItem;
    EditSource: TComboBox;
    PanelFormattingOptions: TPanel;
    LabelFormattingOptions: TLabel;
    optFormatting1: TRadioButton;
    optFormatting2: TRadioButton;
    PanelOptions: TPanel;
    LabelOptions: TLabel;
    cbAddressNewLine: TCheckBox;
    cbExtraComments: TCheckBox;

		procedure FormCreate(Sender: TObject);
		procedure FormDestroy(Sender: TObject);

		procedure mnuFileOpenClick(Sender: TObject);
		procedure mnuFileExitClick(Sender: TObject);
		procedure mnuHelpContentsClick(Sender: TObject);
		procedure mnuHelpAboutClick(Sender: TObject);

		procedure optSizeLimit1Click(Sender: TObject);
		procedure optSizeLimit2Click(Sender: TObject);
		procedure EditDestFolderExit(Sender: TObject);
		procedure EditTargetExtExit(Sender: TObject);
		procedure EditSizeLimitChange(Sender: TObject);
		procedure UpDownSizeLimitClick(Sender: TObject; Button: TUDBtnType);
		procedure cmdBrowseClick(Sender: TObject);
		procedure cmdStartClick(Sender: TObject);
		procedure mnuRegClick(Sender: TObject);
		procedure FormActivate(Sender: TObject);
		procedure optTargetClick(Sender: TObject);
		procedure EditSourceClick(Sender: TObject);
		procedure EditDestFolderKeyPress(Sender: TObject; var Key: Char);
		procedure EditSourceKeyPress(Sender: TObject; var Key: Char);
		procedure FormKeyPress(Sender: TObject; var Key: Char);
		procedure FormWindowProc ( var Message : TMessage );
		procedure HandleDroppedFiles ( var Msg : TMessage );
    procedure optFormattingClick(Sender: TObject);
    procedure cbAddressNewLineClick(Sender: TObject);
    procedure cbExtraCommentsClick(Sender: TObject);
	private

		{ Private declarations }
		DefaultWindowProc: TWndMethod;
		//
		sInitDir : String;
		//
		{ Settings }
		sDestDir : String;
		sTargetExt : String;
		bBreakFile : Boolean;
		iBreakSize : Integer;
		iTargetAsm : Integer;
		iFormatting : Integer;
		bAddressNewLine : Boolean;
		bExtraComments : Boolean;
		//
		nComments : Integer;
		Comments : Array [ 1 .. 100 ] Of TComment;
		//
		ZeroPage : Array [ 0 .. 255 ] Of Word;
		ZeroPageVector : Array [ 0 .. 255 ] Of Boolean;
		Memory : PMemory;
		MemCode : PMemCode;
		MemData : PFlags;
		DataStyles : PDataStyles;
		Labels : PLabels;
		SubParams : PSubParams;
		sPSubParams : String;
		JumpTables : Array [ 1 .. 1000 ] Of TJumpTable;
		nJumpTables : Integer;
		HardStops : PFlags;
		mBeg, mEnd : Integer;
		nStarts : Integer;
		mStarts : Array [ 1 .. 1024 ] Of Integer;
		sStartAddresses : String;

		Procedure ClearAll;
		Procedure AddStart ( i : Integer );
		Function IsStringChar ( b : Byte ) : Boolean;
		Function IsCBMChar ( b : Byte ) : Boolean;
		Function IsJumpTable ( addr : Integer ) : Boolean;
		Function IsInsideJumpTable ( addr : Integer ) : Integer;

	public
		{ Public declarations }
		Function GuessStartAddress : String;
		procedure DisAssemble;
		Function DisAssembleFile ( sFileName : String ) : Boolean;
		procedure Dump ( Cols : Integer );
		procedure DumpAss ( sFileName : String );
		procedure DumpCode;
		procedure RunCode ( StartAddr : Word; lt : TLabelType );
		procedure Settings ( bSave : Boolean );
		procedure CreateDisAssFile ( sFileName : String );
		//
		procedure BrowseFolder ( sFolder : String );
		procedure BrowseFile ( sFilename : String );
	end;

var
	FormMain: TFormMain;

implementation

{$R *.DFM}

procedure TFormMain.FormCreate(Sender: TObject);

Var
	sPath : String;

(**
 * Read the specified file and initialize the commands array
 *)
procedure InitComments ( sFilename : String );
Var
	tf : TextFile;
	s, s2 : String;
	Position, Comma : Integer;
Begin
	nComments := 0;
	If FileExists ( sFilename ) Then
	Begin
		AssignFile ( tf, sFilename );
		Reset ( tf );
		While Not EOF ( tf ) Do
		Begin
			// get a new line
			ReadLn ( tf, s );
			// strip comments
			If Pos ( ';', s ) > 0 Then
				s := Copy ( s, 1, Pos ( ';', s ) - 1 );
			// skip empty lines
			If Trim ( s ) <> '' Then
			Begin
				// is this a new comment definition?
				If Copy ( s, 1, 1 ) = '[' Then
				Begin
					Inc ( nComments );
					s := RemoveFromLeft ( s, 1 );
					If KeepFromRight ( s, 1 ) = ']' Then
						s := Copy ( s, 1, Length ( s ) - 1 );
					s := Trim ( s );
					Comments [ nComments ].Comment := s;
					Comments [ nComments ].nCommentBytes := 0;
				End
				Else
				Begin
					// ignore bytes that are not inside the first [] mark
					If nComments > 0 Then
					Begin
						SplitAtMark ( s, s2, '=' );
						Position := SafeVal ( s );
						//
						Repeat
							Comma := Pos ( ',', s2 );
							If Comma = 0 Then
							Begin
								s := s2;
								s2 := '';
							End
							Else
							Begin
								s := Copy ( s2, 1, Comma - 1 );
								s2 := RemoveFromLeft ( s2, Comma );
							End;
							//
							Inc ( Comments [ nComments ].nCommentBytes );
							Comments [ nComments ].CommentBytes [ Comments [ nComments ].nCommentBytes ].Position := Position;
							Comments [ nComments ].CommentBytes [ Comments [ nComments ].nCommentBytes ].MatchValue := Trim ( s );
							//
							Inc ( Position );
							//
						Until s2 = '';
					End;
				End;
			End;
		End;
	End;
End;

begin
	//
	// Set help file to correct path
	//
	sPath := Copy ( Application.ExeName, 1, PosR ( '\', Application.ExeName ) );
	Application.Helpfile := sPath + 'Disass.hlp>main';
	//
	// Allocate memory for Plus/4 Virtual Memory
	//
	GetMem ( Memory, SizeOf ( TMemory ) );
	GetMem ( MemCode, SizeOf ( TMemCode ) );
	GetMem ( MemData, SizeOf ( TFlags ) );
	GetMem ( DataStyles, SizeOf ( TDataStyles ) );
	GetMem ( Labels, SizeOf ( TLabels ) );
	GetMem ( SubParams, SizeOf ( TSubParams ) );
	GetMem ( HardStops, SizeOf ( TFlags ) );
	//
	ClearAll;

	// Retrieve settings from the Registry
	Settings ( False );

	// init user interface
	UpDownSizeLimit.Position := iBreakSize;
	If bBreakFile Then
	Begin
		optSizeLimit2.Checked := True
	End
	Else
	Begin
		optSizeLimit1.Checked := True;
		optSizeLimit1Click ( Sender );
	End;
  //
	If iTargetAsm = 0 Then
		optTarget1.Checked := True
	Else If iTargetAsm = 1 Then
		optTarget2.Checked := True
	Else If iTargetAsm = 2 Then
		optTarget3.Checked := True;
	//
	If iFormatting = 0 Then
		optFormatting1.Checked := True
	Else If iFormatting = 1 Then
		optFormatting2.Checked := True;
	//
	If bAddressNewLine Then
		cbAddressNewLine.Checked := True;
	If bExtraComments Then
		cbExtraComments.Checked := True;
	//
	EditDestFolder.Text := sDestDir;
	BrowseFolder ( sDestDir );
	//
	EditTargetExt.Text := sTargetExt;
	//
	InitComments ( sPath + 'comments.ini' );
	//
	DefaultWindowProc := WindowProc;
	WindowProc := FormWindowProc;
	DragAcceptFiles ( Handle, True );
end;

procedure TFormMain.FormDestroy(Sender: TObject);
begin
	//
	// Release allocated memory
	//
	FreeMem ( Memory, SizeOf ( TMemory ) );
	FreeMem ( MemCode, SizeOf ( TMemCode ) );
	FreeMem ( MemData, SizeOf ( TFlags ) );
	FreeMem ( DataStyles, SizeOf ( TDataStyles ) );
	FreeMem ( Labels, SizeOf ( TLabels ) );
	FreeMem ( SubParams, SizeOf ( TSubParams ) );
	FreeMem ( HardStops, SizeOf ( TFlags ) );

	// Save settings to the Registry
	Settings ( True );
end;

procedure TFormMain.mnuHelpAboutClick(Sender: TObject);
begin
	//
	// Display about box
	//
	MessageDlg ( 'DisAss - Commodore Plus 4 File DisAssember' + #13 +
		'http://plus4.emucamp.com/tools/disass' + #13 +
		'Version 1.26 (2008/11/15)' + #13 + #13 +
		'Coded by Csabo 2000-2008' + #13 +
		'Write to: plus4@rogers.com', mtInformation, [mbOK], 0 );
end;

procedure TFormMain.mnuHelpContentsClick(Sender: TObject);
begin
	Application.HelpCommand ( HELP_FINDER, 0 );
end;

procedure TFormMain.optSizeLimit1Click(Sender: TObject);
begin
	EditSizeLimit.Enabled := False;
	UpDownSizeLimit.Enabled := False;
	bBreakFile := False;
end;

procedure TFormMain.optSizeLimit2Click(Sender: TObject);
begin
	EditSizeLimit.Enabled := True;
	UpDownSizeLimit.Enabled := True;
	bBreakFile := True;
end;

procedure TFormMain.mnuFileExitClick(Sender: TObject);
begin
	Close;
end;

procedure TFormMain.BrowseFile ( sFilename : String );
Begin
	EditSource.Text := sFileName;
	//
	// Set dest folder to same as source folder
	//
	sFilename := Copy ( sFileName, 1, PosR ( '\', sFileName ) );
	EditDestFolder.Text := sFilename;
	EditDestFolderExit ( Self );
	//
	BrowseFolder ( sFilename );
	//
	cmdStart.SetFocus;
End;

procedure TFormMain.mnuFileOpenClick(Sender: TObject);
begin
	With OpenDialog1 Do
	Begin
		Title := 'Select PRG file';
		InitialDir := sInitDir;
		If Execute Then
		Begin
			BrowseFile ( Filename );
		End;
	End;
end;

procedure TFormMain.BrowseFolder ( sFolder : String );
Var
	sr : TSearchRec;
	iError : Integer;
Begin
	//
	EditSource.Items.Clear;
	//
	If KeepFromRight ( sFolder, 1 ) <> '\' Then
		sFolder := sFolder + '\';
	iError := FindFirst ( sFolder + '*.prg', faAnyFile XOR faDirectory, sr );
	//
	While iError = 0 Do
	Begin
		EditSource.Items.Add ( sr.Name );
		iError := FindNext ( sr );
	End;
End;

procedure TFormMain.Settings ( bSave : Boolean );
var
	r : TRegistry;
begin
	r := TRegistry.Create;
	r.RootKey := HKEY_CURRENT_USER;
	If r.OpenKey ( 'Software\Csabo\DisAss', bSave ) Then
	Begin
		If bSave Then
		Begin
			// Write settings to registry
			r.WriteString ( 'InitDir', sInitDir );
			r.WriteString ( 'DestDir', sDestDir );
			r.WriteString ( 'TargetExt', sTargetExt );
			r.WriteBool ( 'BreakFile', bBreakFile );
			r.WriteInteger ( 'BreakSize', iBreakSize );
			r.WriteInteger ( 'Target', iTargetAsm );
			r.WriteInteger ( 'Formatting', iFormatting );
			r.WriteBool ( 'AddressNewLine', bAddressNewLine );
			r.WriteBool ( 'ExtraComments', bExtraComments );
		End
		Else
		Begin
			// Read settings from registry
			sInitDir := r.ReadString ( 'InitDir' );
			sDestDir := r.ReadString ( 'DestDir' );
			//
			If Pos ( '\', sInitDir ) = 0 Then
			Begin
				sInitDir := sDestDir;
			End;
			//
			sTargetExt := r.ReadString ( 'TargetExt' );
			If r.ValueExists ( 'BreakFile' ) Then
				bBreakFile := r.ReadBool ( 'BreakFile' );
			If r.ValueExists ( 'BreakSize' ) Then
				iBreakSize := r.ReadInteger ( 'BreakSize' );
			If r.ValueExists ( 'Target' ) Then
				iTargetAsm := r.ReadInteger ( 'Target' );
			If r.ValueExists ( 'Formatting' ) Then
				iFormatting := r.ReadInteger ( 'Formatting' );
			If r.ValueExists ( 'AddressNewLine' ) Then
				bAddressNewLine := r.ReadBool ( 'AddressNewLine' );
			If r.ValueExists ( 'ExtraComments' ) Then
				bExtraComments := r.ReadBool ( 'ExtraComments' );
		End;
		r.CloseKey;
	End;
	r.Free;
end;

Procedure TFormMain.ClearAll;
Var
	j : Integer;
Begin
	nStarts := 0;
	//
	For j := 0 To SizeOf ( TMemory ) - 1 Do
		Memory^ [ j ] := 0;

	For j := 0 To SizeOf ( TMemCode ) - 1 Do
		MemCode^ [ j ] := mcNone;

	For j := 0 To SizeOf ( TFlags ) - 1 Do
		MemData^ [ j ] := False;

	For j := 0 To Max - 1 Do
	Begin
		DataStyles^ [ j ].DataFormat := dfAuto;
		DataStyles^ [ j ].Columns := 16;
		DataStyles^ [ j ].StopByte := -1;
	End;

	For j := 0 To SizeOf ( TLabels ) - 1 Do
		Labels [ j ] := ltNone;

	For j := 0 To Max - 1 Do
	Begin
		SubParams [ j ].SpecialType := ssNone;
		SubParams [ j ].Value := 0;
	End;

	For j := 0 To SizeOf ( TFlags ) - 1 Do
		HardStops^ [ j ] := False;

	nJumpTables := 0;

	For j := 0 To 255 Do
	Begin
		ZeroPage [ j ] := 0;
		ZeroPageVector [ j ] := False;
	End;
End;

Procedure TFormMain.AddStart ( i : Integer );
Begin
	//
	// Adds a new start address to the mStarts array
	//
	Inc ( nStarts );
	mStarts [ nStarts ] := i;
End;

Function TFormMain.IsStringChar ( b : Byte ) : Boolean;
Begin
	If ( ( b >= $41 ) And ( b <= $5A ) )
	Or ( ( b >= $61 ) And ( b <= $7A ) )
	Or ( ( b >= $30 ) And ( b <= $3A ) )
	Or ( b = $20 ) Or ( b = $21 ) Or ( b = $27 )
	Or ( b = $2A ) Or ( b = $2D ) Or ( b = $2E ) Or ( b = $2F ) Or ( b = $3F ) Then
		IsStringChar := True
	Else
		IsStringChar := False;
End;

Function TFormMain.IsCBMChar ( b : Byte ) : Boolean;
Begin
	If ( b > $00 ) And ( b < $1B ) Then
		IsCBMChar := True
	Else
		IsCBMChar := False;
End;

Function TFormMain.IsJumpTable ( addr : Integer ) : Boolean;
Var
	i : Integer;
	b : Boolean;
Begin
	b := False;
	i := 1;
	While Not b And ( i <= nJumpTables ) Do
	Begin
		If addr = JumpTables [ i ].Address Then
		Begin
			b := True;
		End
		Else
		Begin
			Inc ( i );
		End;
	End;
	IsJumpTable := b;
End;

Function TFormMain.IsInsideJumpTable ( addr : Integer ) : Integer;
Var
	i : Integer;
	b : Boolean;
Begin
	b := False;
	i := 1;
	While Not b And ( i <= nJumpTables ) Do
	Begin
		If ( addr >= JumpTables [ i ].Address )
		And ( addr <= JumpTables [ i ].Address + JumpTables [ i ].Length - 1 ) Then
		Begin
			b := True;
		End
		Else
		Begin
			Inc ( i );
		End;
	End;
	If b Then
		IsInsideJumpTable := i
	Else
		IsInsideJumpTable := 0;
End;

Function TFormMain.GuessStartAddress : String;
var
	s : String;
	a : Longint;
Begin
	// first guess: if first instruction is a JMP, assume it's start
	If Memory^ [ mBeg ] = $4C Then
	Begin
		s := s + '$' + IntToHex ( mBeg, 4 ) + ',';
	End;
	//
	For a := mBeg To mEnd - 4 Do
	Begin
		If ( Memory^[ a ] = $78 )
		And ( Memory^[ a+1 ] = $8D )
		And ( ( Memory^[ a+2 ] = $3E ) Or ( Memory^[ a+2 ] = $3F ) )
		And ( Memory^[ a+3 ] = $FF ) Then
		Begin
			s := s + '$' + IntToHex( a, 4 ) + ',';
		End;
	End;
	GuessStartAddress := s;
End;

Function TFormMain.DisAssembleFile ( sFileName : String ) : Boolean;
Type
	TDFSection = ( secNone, secStartAddresses, secSubParams,
		secJumpTables,
		secHardStops, secMemoryLayout );
var
	tf : TextFile;
	F : File;
	fs : Integer;
	i, i2, n, a, cols : Integer;
	b : Byte;
	df : TDataFormat;
	//
	s, s0 : String;
	mStart : Integer;
	bDec : Boolean;
	bError : Boolean;
	bDisassFile : Boolean;
	//
	DFSection : TDFSection;
	//
Begin
	//
	ClearAll;
	sStartAddresses := '';
	bError := False;
	mStart := 0;
	bDisassFile := False;
	//
	If Not FileExists ( sFileName ) Then
	Begin
		MessageDlg ( 'File not found.' + #13 + sFileName, mtError, [mbOK], 0 );
		bError := True;
	End
	Else
	Begin
		//
		sInitDir := sFileName;
		If Pos ( '\', sInitDir ) > 0 Then
		Begin
			While sInitDir [ Length ( sInitDir ) ] <> '\' Do
				sInitDir := Copy ( sInitDir, 1, Length ( sInitDir ) - 1 );
		End;
		//
		sStartAddresses := '';
		//
		// --- Check for special disass file
		//
		If FileExists ( sFileName + '.disass' ) Then
		Begin
			//
			// --- Read it
			//
			bDisassFile := True;
			//
			AssignFile ( tf, sFileName + '.disass' );
			Reset ( tf );
			//
			DFSection := secNone;
			While Not Eof ( tf ) Do
			Begin
				ReadLn ( tf, s );
				s := Trim ( s );
				//
				// remove comments
				If Pos ( ';', s ) > 0 Then
				Begin
					s := Copy ( s, 1, Pos ( ';', s ) - 1 );
				End;
				//
				If s <> '' Then
				Begin
					If Copy ( s, 1, 1 ) = '[' Then
					Begin
						DFSection := secNone;
						//
						If Pos ( 'START', UpperCase ( s ) ) > 0 Then
							DFSection := secStartAddresses;
						If Pos ( 'SUB', UpperCase ( s ) ) > 0 Then
							DFSection := secSubParams;
						If Pos ( 'HARD', UpperCase ( s ) ) > 0 Then
							DFSection := secHardStops;
						If Pos ( 'JUMP', UpperCase ( s ) ) > 0 Then
							DFSection := secJumpTables;
						If Pos ( 'MEMORY', UpperCase ( s ) ) > 0 Then
							DFSection := secMemoryLayout;
					End
					Else
					Begin
						//
						Case DFSection Of

							secStartAddresses :
							Begin
								sStartAddresses := s;
							End;

							secSubParams :
							Begin
								i := SafeVal ( Copy ( s, 1, 5 ) );
								//
								s := Trim ( RemoveFromLeft ( s, 5 ) );
								If s [ 1 ] = ',' Then
								Begin
									s := UpperCase ( Trim ( RemoveFromLeft ( s, 1 ) ) );
								End;
								If s [ 1 ] = 'L' Then
									SubParams [ i ].SpecialType := ssLength
								Else
									SubParams [ i ].SpecialType := ssStopByte;
								//
								If Pos ( ',', s ) > 0 Then
								Begin
									s := Trim ( RemoveFromLeft ( s, Pos ( ',', s ) ) );
								End
								Else
								Begin
									s := Trim ( RemoveFromLeft ( s, Pos ( ' ', s ) ) );
								End;
								SubParams [ i ].Value := SafeVal ( s );
							End;

							secHardStops :
							Begin
								i := SafeVal ( Copy ( s, 1, 5 ) );
								//
								HardStops [ i ] := True;
							End;

							secJumpTables :
							Begin
								s := Replace ( s, ',', ' ' );
								//
								i := SafeVal ( Copy ( s, 1, 5 ) );
								s := Trim ( RemoveFromLeft ( s, 5 ) );
								b := SafeVal ( Copy ( s, 1, Pos ( ' ', s ) - 1 ) );
								s := Trim ( RemoveFromLeft ( s, Pos ( ' ', s ) ) );
								Inc ( nJumpTables );
								With JumpTables [ nJumpTables ] Do
								Begin
									Address := i;
									Length := b;
									Style := SafeVal ( s );
								End;
							End;

							secMemoryLayout :
							Begin
								s := Replace ( s, ',', ' ' );
								//
								i := SafeVal ( Copy ( s, 1, 5 ) );
								i2 := SafeVal ( Copy ( s, Pos ( '-', s ) + 1, 5 ) );
								//
								s := UpperCase ( Trim ( RemoveFromLeft ( s, Pos ( ' ', s ) ) ) );
								//
								df := dfAuto;
								If ( Length ( s ) > 0 ) And ( s [ 1 ] = 'W' ) Then
								Begin
									df := dfWord;
								End;
								If ( Length ( s ) > 0 ) And ( s [ 1 ] = 'B' ) Then
								Begin
									df := dfByte;
								End;
								//
								s := Trim ( RemoveFromLeft ( s, Pos ( ' ', s ) ) );
								cols := SafeVal ( Copy ( s, 1, Pos ( ' ', s ) - 1 ) ); // columns
								If cols <= 0 Then cols := 16;
								//
								s := Trim ( RemoveFromLeft ( s, Pos ( ' ', s ) ) );
								a := SafeVal ( s ); // stopbyte
								//
								While ( i <= i2 ) Do
								Begin
									//
									With DataStyles [ i ] Do
									Begin
										DataFormat := df;
										Columns := cols;
										StopByte := a;
									End;
									//
									Inc ( i );
								End;
							End;

						End;
					End;
				End;
			End;
			//
			CloseFile ( tf );
		End;
		//
		AssignFile ( F, sFileName );
		Reset ( F, 1 );
		fs := FileSize ( F );
		// Check size
		If fs - 2 > SizeOf ( TMemory ) Then
		Begin
			// Won't fit into memory
			CloseFile ( F );
			MessageDlg ( 'File is too large.' + #13 +
				'(' + IntToStr ( fs Div 1024 ) + 'KB)', mtError, [mbOK], 0 );
			bError := True;
		End
		Else
		Begin
			// Read start address
			BlockRead ( F, mBeg, 2 );
			//
			If ( mBeg + fs - 2 ) > SizeOf ( TMemory ) Then
			Begin
				// Won't fit into memory
				CloseFile ( F );
				MessageDlg ( 'File will not fit into 64K.' + #13 +
					'(Start Address: $' + IntToHex ( mBeg, 4 ) +
					' File size:' + IntToStr ( fs Div 1024 ) + 'KB)', mtError, [mbOK], 0 );
				bError := True;
			End
			Else
			Begin
				//
				// *** Load actual program ***
				//
				BlockRead ( F, Memory^ [ mBeg ], fs - 2 );
				CloseFile ( F );
				mEnd := mBeg + fs - 2;
				//
				//
				//
				s := '';
				//
				// *** Look for SYS basic token
				//
				i := $1005;
				While ( i < $1080 ) And ( Memory^ [ i ] <> $9E ) Do
				Begin
					Inc ( i );
				End;
				If Memory^ [ i ] = $9E Then
				Begin
					//
					// Present, read sys number
					//
					mStart := 0;
					Inc ( i );
					b := Memory^ [ i ];
					bDec := False;
					//
					While ( ( b >= 48 ) And ( b <= 57 ) )
					Or ( b = $28 ) Or ( b = 34 ) Or ( b = $20 ) Or ( b = 209 ) Do
					Begin
						// ignore space, opening bracket and "
						If ( b >= 48 ) And ( b <= 57 ) Then
						Begin
							If bDec Then
							Begin
								mStart := mStart * 16 + b - 48;
							End
							Else
							Begin
								mStart := mStart * 10 + b - 48;
							End;
						End
						Else
						Begin
							If b = 209 Then
								bDec := True; // basic token for DEC("####") command
						End;
						Inc ( i );
						b := Memory^ [ i ];
					End;
					//
					s := s + '$' + IntToHex ( mStart, 4 ) + ',';
				End;
				//
				// *** Special check for ROM files
				//     if load address is $4002,$8000 or $C000
				//     then use last 3 vectors as start addresses
				//
				If ( ( mBeg = $8000 ) And ( fs = $8002 ) )
				Or ( ( mBeg = $C000 ) And ( fs = $4002 ) ) Then
				Begin
					s := s +
						'$' + IntToHex ( Memory [ $FFFA ] + Memory [ $FFFB ] Shl 8, 4 ) + ',' +
						'$' + IntToHex ( Memory [ $FFFC ] + Memory [ $FFFD ] Shl 8, 4 ) + ',' +
						'$' + IntToHex ( Memory [ $FFFE ] + Memory [ $FFFF ] Shl 8, 4 ) + ',';
				End;
				//
				// *** Check for start address in file name
				//
				s0 := sFileName;
				s0 := Copy ( s0, 1, PosR ( '.', s0 ) - 1 );
				s0 := RemoveFromLeft ( s0, PosR ( '\', s0 ) );
				//
				// Check for G#### in filename
				If ( UpperCase ( Copy ( KeepFromRight ( s0, 5 ), 1, 1 ) ) = 'G' )
				And ( SafeVal ( '$' + KeepFromRight ( s0, 4 ) ) > 0 ) Then
				Begin
					s := s + '$' + UpperCase ( KeepFromRight ( s0, 4 ) ) + ',';
				End
				Else
				Begin
					// Check for SYS#### in filename
					If ( UpperCase ( Copy ( KeepFromRight ( s0, 7 ), 1, 3 ) ) = 'SYS' )
					And ( SafeVal ( '$' + KeepFromRight ( s0, 4 ) ) > 0 ) Then
					Begin
						s := s + KeepFromRight ( s0, 4 ) + ',';
					End
					Else
					Begin
						// Check for SYS##### in filename
						If ( UpperCase ( Copy ( KeepFromRight ( s0, 8 ), 1, 3 ) ) = 'SYS' )
						And ( SafeVal ( '$' + KeepFromRight ( s0, 5 ) ) > 0 ) Then
						Begin
							s := s + KeepFromRight ( s0, 5 ) + ',';
						End;
					End;
				End;
				//
				// *** Last resort: see if first instruction is a jump.
				//
				if ( s = '' ) Then
				Begin
					s := GuessStartAddress;
				End;
				//
				If s <> '' Then
				Begin
					// remove last comma + space
					s := RemoveFromRight ( s, 1 );
				End
				Else
				Begin
					// No luck... unknown start address...
					s := '(start address not known)';
				End;
				//
				// ask anyway
				//
				if sStartAddresses = '(start address not known)' Then
					sStartAddresses := '';
				if sStartAddresses <> '' Then
					s := sStartAddresses;
				//
				s0 := 'Start Address';
				If bDisassFile Then
					s0 := s0 + ' (Disass file found)';
				s := InputBox ( s0, 'Enter the start addresses separated by commas:', s);
				sStartAddresses := s;
				//
				While s <> '' Do
				Begin
					If Pos ( ',', s ) > 0 Then
					Begin
						mStart := SafeVal ( Copy ( s, 1, Pos ( ',', s ) - 1 ) );
						s := RemoveFromLeft ( s, Pos ( ',', s ) );
					End
					Else
					Begin
						mStart := SafeVal ( s );
						s := '';
					End;
					AddStart ( mStart );
				End;
				//
				s := sFileName;
				s := Copy ( s, 1, PosR ( '.', s ) - 1 );
				s := RemoveFromLeft ( s, PosR ( '\', s ) );
				EditProgramName.Text := s;
				//
				EditMemBeg.Text := '$' + IntToHex ( mBeg, 4 );
				EditMemEnd.Text := '$' + IntToHex ( mEnd, 4 );
				EditMemStart.Text := '$' + IntToHex ( mStart, 4 );
			End;
		End;
	End;

	If Not bError Then
	Begin
		//
		// Analyze
		//
		If ( nStarts = 0 )
		Or ( ( nStarts = 1 ) And ( mStarts [ 1 ] = 0 ) ) Then
		Begin
			// run everything!
			a := mBeg;
			While ( a < mEnd ) Do
			Begin
				If ( MemCode^ [ a ] <> mcInstruction )
				And ( MemCode^ [ a ] <> mcParams )
				And ( Memory^ [ a ] <> 0 ) Then
					RunCode ( a, ltNone );
				Inc ( a );
			End;
		End
		Else
		Begin
			For i := 1 To nStarts Do
			Begin
				RunCode ( mStarts [ i ], ltStart );
			End;
		End;
		//
		// Deal with Jump Tables
		//
		For i := 1 To nJumpTables Do
		Begin
			n := 0;
			While n < JumpTables [ i ].Length Do
			Begin
				Case JumpTables [ i ].Style Of
					//
					0 : // address is lo,hi; lo,hi; ...
					Begin
						a := Memory^ [ JumpTables [ i ].Address + n ] +
							Memory^ [ JumpTables [ i ].Address + n + 1 ] Shl 8;
						RunCode ( a, ltJump );
						Inc ( n, 2 );
					End;
					//
					1 : // address is lo+1,hi; lo+1,hi; ...
					Begin
						a := Memory^ [ JumpTables [ i ].Address + n ] +
							Memory^ [ JumpTables [ i ].Address + n + 1 ] Shl 8;
						RunCode ( a + 1, ltJump );
						Inc ( n, 2 );
					End;
					//
					2 :
					Begin
						a := Memory^ [ JumpTables [ i ].Address + n ] +
							Memory^ [ JumpTables [ i ].Address + JumpTables [ i ].Length Div 2 + n ] Shl 8;
						RunCode ( a, ltJump );
						Inc ( n );
						// stop loop at half of table
						If n = ( JumpTables [ i ].Length Div 2 ) Then
						Begin
							n := JumpTables [ i ].Length;
						End;
					End;
					//
					3 : // address is hi,lo; hi,lo; ...
					Begin
						a := Memory^ [ JumpTables [ i ].Address + n + 1 ] +
							Memory^ [ JumpTables [ i ].Address + n ] Shl 8;
						RunCode ( a, ltJump );
						Inc ( n, 2 );
					End;
					//
					4 :
					Begin
						a := Memory^ [ JumpTables [ i ].Address + n ] Shl 8 +
							Memory^ [ JumpTables [ i ].Address + JumpTables [ i ].Length Div 2 + n ];
						RunCode ( a, ltJump );
						Inc ( n );
						// stop loop at half of table
						If n = ( JumpTables [ i ].Length Div 2 ) Then
						Begin
							n := JumpTables [ i ].Length;
						End;
					End;
					//
				End;
			End;
		End;
		//
		// Stats
		//
		DumpCode;
		//
		// Output
		//
		DumpAss ( sFileName );
		//
		CreateDisAssFile ( sFileName );
		//
	End;
	//
	DisAssembleFile := bError;
end;

// unused procedure
procedure TFormMain.Dump ( Cols : Integer );
var
	i : Integer;
	s : String;
begin
	For i := $1000 To $1100 Do
	Begin
		If ( i Mod Cols ) = 0 Then
		Begin
			s := '>' + IntToHex ( i, 4 );
		End;
		s := s + ' '  + IntToHex ( Memory^[i], 2 );
		If ( i Mod Cols ) = Cols - 1 Then
		Begin
			MemoPreview.Lines.Add ( s );
		End;
	End;
end;

Procedure TFormMain.CreateDisAssFile ( sFileName : String );
Var
	tf : TextFile;
	s, i : Integer;
	bData : Boolean;
	l : String;
	//
	prev_df : TDataFormat;
	prev_sb : Integer;
Begin
	AssignFile ( tf, sFileName + '.disass' );
	ReWrite ( tf );
	//
	WriteLn ( tf, '; This file was generated by DisAss.' );
	WriteLn ( tf, '; Feel free to modify it.' );
	WriteLn ( tf );
	//
	WriteLn ( tf, '[Start Addresses]' );
	WriteLn ( tf, sStartAddresses );
	WriteLn ( tf );
	//
	WriteLn ( tf, '[SubParams]' );
	WriteLn ( tf, '; Format: "$ADDR Length/StopByte <VALUE>"' );
	For i := 0 To $FFFF Do
	Begin
		If SubParams^ [ i ].SpecialType <> ssNone Then
		Begin
			Write ( tf, '$' + IntToHex ( i, 4 ) + ', ' );
			If SubParams^ [ i ].SpecialType = ssLength Then
				Write ( tf, 'Length, ' )
			Else
				Write ( tf, 'StopByte, ' );
			WriteLn ( tf, IntToStr ( SubParams^ [ i ].Value ) );
		End;
	End;
	WriteLn ( tf, sPSubParams ); // includes cr/lf at end
	//WriteLn ( tf );
	//
	WriteLn ( tf, '[JumpTables]' );
	WriteLn ( tf, '; Format: "$ADDR <LENGTH_OF_TABLE> <STYLE>"' );
	WriteLn ( tf, '; Styles: 0: LO,HI;... 1: LO+1,HI;... 2: LO,LO... HI,HI... 3: HI,LO;... 4:HI,HI... LO,LO..."' );
	For i := 1 To nJumpTables Do
	Begin
		WriteLn ( tf, '$' + IntToHex ( JumpTables [ i ].Address, 4 ) + ', ' +
			IntToStr ( JumpTables [ i ].Length ) + ', ' +
			IntToStr ( JumpTables [ i ].Style ) );
	End;
	WriteLn ( tf );
	//
	WriteLn ( tf, '[HardStops]' );
	WriteLn ( tf, '; Format: "$ADDR"' );
	For i := 0 To $FFFF Do
	Begin
		If HardStops^ [ i ] Then
		Begin
			WriteLn ( tf, '$' + IntToHex ( i, 4 ) );
		End;
	End;
	WriteLn ( tf );
	//
	WriteLn ( tf, '[Memory Layout]' );
	WriteLn ( tf, '; Format: "$ADDR-$ADDR Byte/Word <StopByte>"' );
	//
	s := mBeg;
	bData := MemCode^ [ s ] = mcNone;
	//
	prev_df := Succ ( DataStyles [ mBeg ].DataFormat );
	prev_sb := ( DataStyles [ mBeg ].StopByte + 1 ) And 255;
	//
	For i := mBeg To mEnd - 1 Do
	Begin
		If i > mBeg Then
		Begin
			If ( i = mEnd - 1 )
			Or ( bData And ( MemCode^ [ i ] <> mcNone ) )
			Or ( Not bData And ( MemCode^ [ i ] = mcNone ) )
			Or ( prev_df <> DataStyles [ i ].DataFormat )
			Or ( prev_sb <> DataStyles [ i ].StopByte ) Then
			Begin
				l := '$' + IntToHex ( s, 4 ) + '-$' + IntToHex ( i - 1, 4 ) + ' ';
				Case DataStyles [ i - 1 ].DataFormat Of
					dfAuto : l := l + 'AUTO';
					dfByte : l := l + 'BYTE';
					dfWord : l := l + 'WORD';
				End;
				//
				l := l + ' ' + IntToStr ( DataStyles [ i - 1 ].Columns );
				l := l + ' ' + IntToStr ( DataStyles [ i - 1 ].StopByte );
				//
				l := l + ' ; ' + IntToHex ( i - s, 4 ) + ' bytes of ';
				If bData Then
					l := l + 'data'
				Else
					l := l + 'code';
				//
				WriteLn ( tf, l );
				s := i;
				//
				bData := MemCode^ [ i ] = mcNone;
			End;
		End;
		//
		prev_df := DataStyles [ i ].DataFormat;
		prev_sb := DataStyles [ i ].StopByte;
	End;
	//
	CloseFile ( tf );
End;

// Generate the output files
procedure TFormMain.DumpAss ( sFileName : String );
var
	a, Count, i : Integer;
	OpCode, ad, by : Byte;
	rel : Shortint;
	value : Byte;
	dest : Word;
	s, sComm : String;
	Bits, KeyColumn : Integer;
	bStringOpen : Boolean; // if true, we're putting bytes as "abcd" etc.
	bStart : Boolean;
	bPrevOpcode : Boolean; // if true, the previous line was an opcode (not data)

	DataBytes : Integer;
	bWordMode, bLineWordMode : Boolean;

	OutF : TextFile;
	OutF_Open : Boolean;
	Cnt, FileCount : Integer;
	FSize : Integer;

	iComment, iCommentBytes : Integer;
	bCommentMatch : Boolean;
	bCommentByte, bCommentPos : Integer;

procedure destComment ( a16 : Word; Var s : String );
Const
	TedRegs : Array [ 0 .. $1F ] Of String =
	(
		'Timer 1 Lo',
		'Timer 1 Hi',
		'Timer 2 Lo',
		'Timer 2 Hi',
		'Timer 3 Lo',
		'Timer 3 Hi',
		'Vert Scroll',
		'Horiz Scroll',
		'Keyboard Latch',
		'IRQ',
		'IRQ req',
		'Vert raster',
		'Cursor C',
		'Cursor D',
		'Sound1 Freq Lo',
		'Sound2 Freq Lo',
		'Sound2 Freq Hi',
		'Volume',
		'Sound1 Freq Hi / Gfx',
		'Char. Gen',
		'Video Matrix',
		'Background color',
		'Multi Color 1',
		'Multi Color 2',
		'Multi Color 3',
		'Border Color',
		'1A',
		'1B',
		'1C',
		'Vert Scan',
		'Horiz Scan',
		'Vert. Raster / Flash'
	);

Type
	TSpecialAddress = Record
		a : Word;
		d : String;
	End;

Const
	SpecialAddressesMax = 58;
	SpecialAddresses : Array [ 1 .. SpecialAddressesMax ] Of TSpecialAddress =
	(
		( a : $0312; d : 'Vector Lo Byte' ),
		( a : $0313; d : 'Vector Hi Byte' ),

		( a : $0314; d : 'IRQ Vector Lo Byte' ),
		( a : $0315; d : 'IRQ Vector Hi Byte' ),

		( a : $FCF1; d : 'Banking: JMP to cartridge IRQ routine' ),
		( a : $FCF4; d : 'Banking: JMP to PHOENIX routine' ),
		( a : $FCF7; d : 'Banking: JMP to LONG FETCH routine' ),
		( a : $FCFA; d : 'Banking: JMP to LONG JUMP routine' ),
		( a : $FCFD; d : 'Banking: JMP to LONG IRQ routine' ),

		( a : $FF49; d : 'JMP to define function key routine' ),
		( a : $FF4C; d : 'JMP to PRINT routine' ),
		( a : $FF4F; d : 'JMP to PRIMM routine' ),
		( a : $FF52; d : 'JMP to ENTRY routine' ),
		( a : $FF80; d : 'RELEASE # OF KERNAL (MSB: 0 = NTSC ; 1 = PAL)' ),

		( a : $FF81; d : 'KERNAL: CINT (Initialize screen editor)' ),
		( a : $FF84; d : 'KERNAL: IOINIT (Initialize I/O devices)' ),
		( a : $FF87; d : 'KERNAL: RAMTAS (Ram test)' ),
		( a : $FF8A; d : 'KERNAL: RESTOR (Restore vectors to initial values)' ),
		( a : $FF8D; d : 'KERNAL: VECTOR (Change vectors for user)' ),
		( a : $FF90; d : 'KERNAL: SETMSG (Control O.S. messages)' ),
		( a : $FF93; d : 'KERNAL: SECND (Send SA after LISTEN)' ),
		( a : $FF96; d : 'KERNAL: TKSA (Send SA after TALK)' ),
		( a : $FF99; d : 'KERNAL: MEMTOP (Set/Read top of memory)' ),
		( a : $FF9C; d : 'KERNAL: MEMBOT (Set/Read bottom of memory)' ),
		( a : $FF9F; d : 'KERNAL: SCNKEY (Scan keyboard)' ),
		( a : $FFA2; d : 'KERNAL: SETTMO (Set timeout in DMA disk)' ),
		( a : $FFA5; d : 'KERNAL: ACPTR (Handshake serial bus or DMA disk byte in)' ),
		( a : $FFA8; d : 'KERNAL: CIOUT (Handshake serial bus or DMA disk byte out)' ),
		( a : $FFAB; d : 'KERNAL: UNTLR (Send UNTALK out serial bus or DMA disk)' ),
		( a : $FFAE; d : 'KERNAL: UNLSN (Send UNLISTEN out serial bus or DMA disk)' ),
		( a : $FFB1; d : 'KERNAL: LISTN (Send LISTEN out serial bus or DMA disk)' ),
		( a : $FFB4; d : 'KERNAL: TALK (Send TALK out serial bus or DMA disk)' ),
		( a : $FFB7; d : 'KERNAL: READSS (Return I/O STATUS byte)' ),
		( a : $FFBA; d : 'KERNAL: SETLFS (Set LA, FA, SA)' ),
		( a : $FFBD; d : 'KERNAL: SETNAM (Set length and filename address)' ),
		( a : $FFC0; d : 'KERNAL: OPEN (Open logical file)' ),
		( a : $FFC3; d : 'KERNAL: CLOSE (Close logical file)' ),
		( a : $FFC6; d : 'KERNAL: CHKIN (Open channel in)' ),
		( a : $FEC9; d : 'KERNAL: CHOUT (Open channel out)' ),
		( a : $FECC; d : 'KERNAL: CLRCH (Close I/O channels)' ),
		( a : $FFCF; d : 'KERNAL: BASIN (Input from channel)' ),
		( a : $FFD2; d : 'KERNAL: BSOUT (Output to channel)' ),
		( a : $FFD5; d : 'KERNAL: LOADSP (Load from file)' ),
		( a : $FFD8; d : 'KERNAL: SAVESP (Save to file)' ),
		( a : $FFDB; d : 'KERNAL: SETTIM (Set internal clock)' ),
		( a : $FFDE; d : 'KERNAL: RDTIM (Read internal clock)' ),
		( a : $FFE1; d : 'KERNAL: STOP (Scan STOP key)' ),
		( a : $FFE4; d : 'KERNAL: GETIN (Get character from queue)' ),
		( a : $FFE7; d : 'KERNAL: CLALL (Close all files)' ),
		( a : $FFEA; d : 'KERNAL: UDTIM (Increment clock)' ),
		( a : $FFED; d : 'KERNAL: SCRORG (Query screen width/height)' ),
		( a : $FFF0; d : 'KERNAL: PLOT (Read/Set X,Y coord of cursor)' ),
		( a : $FFF3; d : 'KERNAL: IOBASE (Return location of start of I/O)' ),

		( a : $FD30; d : 'Keyboard Matrix' ),

		( a : $FFFC; d : 'Reset Vector Lo Byte' ),
		( a : $FFFD; d : 'Reset Vector Hi Byte' ),
		( a : $FFFE; d : 'NMI Vector Lo Byte' ),
		( a : $FFFF; d : 'NMI Vector Hi Byte' )
	);

Var
	SpecialAddressesCounter : Integer;
	SpecialAddressFound : Boolean;

begin
	// Ted Register?
	If ( a16 >= $FF00 ) And ( a16 <= $FF1F ) Then
	Begin
		// Yes, add apropriate comment
		s := s + #9 + '; ' + TedRegs [ a16 - $FF00 ];
	End
	Else
	Begin
		// Special address?
		SpecialAddressesCounter := 1;
		SpecialAddressFound := False;
		While ( SpecialAddressesCounter <= SpecialAddressesMax )
		And Not SpecialAddressFound Do
		Begin
			If SpecialAddresses [ SpecialAddressesCounter ].a = a16 Then
			Begin
				s := s + #9 + '; ' + SpecialAddresses [ SpecialAddressesCounter ].d;
				SpecialAddressFound := True;
			End
			Else
				Inc ( SpecialAddressesCounter );
		End;
		//
		If Not SpecialAddressFound Then
		Begin
			//
			If ( a16 >= $0800 ) And ( a16 <= $0800 + 25 * 40 ) Then
			Begin
				s := s + #9 + '; screen colour mem';
			End
			Else
			Begin
				If ( a16 >= $0C00 ) And ( a16 <= $0C00 + 25 * 40 ) Then
				Begin
					s := s + #9 + '; screen char mem';
				End
			End;
			//
		End;
	End;
end;

Function MatchCommentByte ( b : Byte; MatchString : String ) : Boolean;

Function MatchByte ( b : Byte; Match : String ) : Boolean;
Begin
	If ( Copy ( Match, 1, 2 ) = '!=' ) Or ( Copy ( Match, 1, 2 ) = '<>' ) Then
	Begin
		MatchByte := b <> SafeVal ( RemoveFromLeft ( Match, 2 ) );
	End
	Else
	Begin
		MatchByte := b = SafeVal ( Match );
	End;
End;

Var
	bMatch : Boolean;
	i : Integer;
	sOne : String;
Begin
	bMatch := False;
	//
	If MatchString = '*' Then
	Begin
		bMatch := True;
	End
	Else
	Begin
		If Pos ( '|', MatchString ) > 0 Then
		Begin
			MatchString := MatchString + '|';
			While MatchString <> '' Do
			Begin
				i := Pos ( '|', MatchString );
				If i = 0 Then
					i := Length ( MatchString ) + 1;
				sOne := Copy ( MatchString, 1, i - 1);
				MatchString := RemoveFromLeft ( MatchString, i );
				//
				bMatch := bMatch Or MatchByte ( b, sOne );
			End;
		End
		Else
		Begin
			bMatch := MatchByte ( b, MatchString );
		End;
	End;
	MatchCommentByte := bMatch;
End;

Function ProcessComment ( s : String; a : Word ) : String;
Var
	i, e : Integer;
	sNumber : String;
Begin
	i := Pos ( '%', s ); // find a '%' sign in the string (special marker)
	While i > 0 Do
	Begin
		e := Pos ( '%', RemoveFromLeft ( s, i ) ); // find the closing pair
		If ( e = 0 ) Then
		Begin
			i := 0;
		End
		Else
		Begin
			sNumber := Copy ( s, i + 1, e - 1 ); // the string that's inbetween is the index of the byte to get
			s := Copy ( s, 1, i - 1 ) + IntToHex ( Memory^[ a + SafeVal ( sNumber ) ], 2 ) + RemoveFromLeft ( s, i + e );
			//
			i := Pos ( '%', s );
		End;
	End;
	//
	ProcessComment := s;
End;

function destAddr ( a16 : Word ) : String;
var
	lt : TLabelType;
	b : Integer;
	s : String;
begin
	If ( ( a16 >= $0100 ) And ( a16 <= $FD00 ) )
	Or ( ( a16 >= $FF40 ) ) Then
	Begin
		//
		//
		//
		{
		if a16 = $1f71 then
		begin
			showmessage('debug breakpoint - DestAddr');
		end;
		}
		//
		b := 0;
		if ( a16 < $FFF0 ) Then
		Begin
			If ( Labels^ [ a16 + b ] = ltNone ) And Not ( MemData^ [ a16 + b ] ) Then
			Begin
				If ( a16 > 0 ) And IsJumpTable ( a16 - 1 ) Then
				Begin
					b := -1;
				End
				Else
				Begin
					While ( Labels^ [ a16 + b ] = ltNone ) And Not ( MemData^ [ a16 + b ] )
					And ( b < 3 ) Do
					Begin
						Inc ( b );
					End;
				End;
			End;
		End;
		//
		If b <> 0 Then
		Begin
			If ( Memory^ [ a16 ] = $4C ) Or ( Memory^ [ a16 ] = $20 ) Then
			Begin
				sComm := '*** Self mod. indirect jump!';
			End;
		End
		Else
		Begin
      //
			// Always point to beginning of jump tables
      //
      If a16 > 0 Then
      Begin
				If IsJumpTable ( a16 - 1 ) Then
					b := 1;
			End;
    End;

		lt := Labels^ [ a16 + b ];

		Case lt Of

			ltJump : s := 'L_';
			ltSub : s := 'S_';
			ltIRQ : s := 'I_';
			ltRes : s := 'R_';
			ltNMI : s := 'N_';
			ltBranch : s := 'B_';
			ltCharset : s := 'C_';
			ltStart : s := 'X_';
      ltVec : s := 'W_';
			Else
			Begin
				If MemData^ [ a16 + b ] Then
				Begin
					s := 'V_';
				End
				Else
				Begin
					s := '$';
				End;
			End;
		End;

		If ( b >= 0 ) Then
		Begin
			b := 0;
			If MemCode^ [ a16 ] = mcParams Then
			Begin
				While ( MemCode^ [ a16 - b ] = mcParams ) Do
				Begin
					Inc ( b );
				End;
				//
				If ( Memory^ [ a16 - b ] = $24 )
				Or ( Memory^ [ a16 - b ] = $2C ) Then
				Begin
					b := 0; // don't offset for BIT $00 and BIT $0000
				End;
			End;
		End
		Else
		Begin
			b := -b;
		End;
		//
		s := s + IntToHex ( a16 - b, 4 );
		If b <> 0 Then
			s := s + ' + ' + IntToStr ( b );
		//
		destAddr := s;
		//
	End
	Else
	Begin

		destAddr := '$' + IntToHex ( a16, 4 );

	End;
end;

Function ZeroPageAddress ( a16 : Byte ) : String;
Begin
	If a16 > 0 Then
	Begin
		If ZeroPageVector [ a16 - 1 ] Then
		Begin
			ZeroPageAddress := 'zp' + IntToHex ( a16 - 1, 2 ) + '+1';
		End
		Else
		Begin
			ZeroPageAddress := 'zp' + IntToHex ( a16, 2 );
		End;
	End
	Else
	Begin
		ZeroPageAddress := 'zp' + IntToHex ( a16, 2 );
	End;
End;

function FindLabel ( b : Byte; Hi : Boolean; LabelType : TLabelType ) : Word;
Var
	a0, aDest : Word;
	Found : Integer;
Begin
	//
	Found := 0;
	aDest := 0;
	//
	If Hi Then
	Begin
		For a0 := 0 To 255 Do
		Begin
			If Labels^ [ b Shl 8 + a0 ] = LabelType Then
			Begin
				Inc ( Found );
				aDest := b Shl 8 + a0;
			End;
		End;
	End
	Else
	Begin
		For a0 := 0 To 255 Do
		Begin
			If Labels^ [ a0 Shl 8 + b ] = LabelType Then
			Begin
				Inc ( Found );
				aDest := a0 Shl 8 + b;
			End;
		End;
	End;
	//
	If Found = 1 Then
		FindLabel := aDest
	Else
		FindLabel := 0;
End;

function FindIRQ ( b : Byte; Hi : Boolean ) : Word;
Begin
	FindIRQ := FindLabel ( b, Hi, ltIRQ );
End;

function FindVec ( b : Byte; Hi : Boolean ) : Word;
Begin
	FindVec := FindLabel ( b, Hi, ltVec );
End;

function FindNMI ( b : Byte; Hi : Boolean ) : Word;
Begin
	FindNMI := FindLabel ( b, Hi, ltNMI );
End;

function LoByteStr ( s : String ) : String;
begin
	Case iTargetAsm Of

		2 : LoByteStr := '<' + s;

		0, 1 : LoByteStr := s + ' & 255';

	End;
end;

function HiByteStr ( s : String ) : String;
begin
	Case iTargetAsm Of

		2 : HiByteStr := '>' + s;

		0, 1 : HiByteStr := s + ' >> 8';

	End;
end;

// LabelStr
// produces appropriate label
// according to target assembler
function LabelStr ( s : String ) : String;
begin
	Case iTargetAsm Of

		2 : LabelStr := s + ':';

		1 : LabelStr := s;

		0 : LabelStr := s;

	End;
end;

// Output one line
procedure Send ( anyString : String );
Var
	sPath, sExt : String;
	sFName : String;
	s : String;
begin
	if anyString <> '' Then
	Begin
		MemoPreview.Lines.Add ( anyString );

		//
		If Not OutF_Open Then
		Begin
			//
			// --- Open a new file
			//
			Inc ( FileCount );

			If sDestDir = '' Then
				sPath := sInitDir
			Else
				sPath := sDestDir;

			If sPath [ Length ( sPath ) ] <> '\' Then
				sPath := sPath + '\';
			sExt := sTargetExt;
			if sExt <> '' Then
			Begin
				If sExt [ 1 ] <> '.' Then
					sExt := '.' + sExt;
			End;
			//
			If bBreakFile Then
			Begin
				sFName := IntToStr ( FileCount );
			End
			Else
			Begin
				sFName := RemoveFromLeft ( sFileName, PosR ( '\', sFileName ) );
				sFName := Copy ( sFName, 1, Pos ( '.', sFName ) - 1 );
			End;
			//
			s := sPath + sFName + sExt;
			//ShowMessage ( s );
			//
			AssignFile ( OutF, s );
			ReWrite ( OutF );
			//
			OutF_Open := True;
		End;
		WriteLn ( OutF, anyString );
		Inc ( FSize, Length ( anyString ) + 2 );

		If bBreakFile And ( FSize > iBreakSize * 1024 ) Then
		Begin

			WriteLn ( OutF, '' );
			WriteLn ( OutF, '#include "' + IntToStr ( FileCount + 1 ) + sExt + '"' );

			CloseFile ( OutF );
			OutF_Open := False;
			FSize := 0;
		End;
	End;
end;

// closes the current 'data' string (if it's open)
Procedure CloseString ( sEnd1 : String );
Begin
	If bStringOpen Then
	Begin
		s := s + '"' + sEnd1;
		bStringOpen := False;
	End;
End;

Procedure FlushData;
Begin
	CloseString ( '' );
	Send ( s );
	//
	DataBytes := 0;
	s := '';
End;

Procedure GetLabel;
Var
	StartLoop : Integer;
Begin
	//
	// Does this instruction need a label?
	//
	If ( a < $FFFE )
	And ( ( Labels^ [ a ] <> ltNone )
	Or ( ( by > 1 ) And ( Labels^ [ a + 1 ] <> ltNone ) )
	Or ( ( by > 2 ) And ( Labels^ [ a + 2 ] <> ltNone ) )
	Or ( MemData^ [ a ] )
	Or ( ( by > 1 ) And ( MemData^ [ a + 1 ] ) )
	Or ( ( by > 2 ) And ( MemData^ [ a + 2 ] ) ) ) Then
	Begin
		bStart := False;
		If ( Labels^ [ a ] = ltStart ) Then
			bStart := True;
		For StartLoop := 1 To nStarts Do
		Begin
			If a = mStarts [ StartLoop ] Then
			Begin
				bStart := True;
			End;
		End;
		//
		If bStart Then
		Begin
			Send ( #9 + ';' );
			Send ( #9 + '; *** Start ***' );
			Send ( #9 + ';' );
		End;
		s := LabelStr ( destAddr ( a ) ) + #9;
	End
	Else
	Begin
		s := s + #9;
	End;
End;

function formatCase( s : String ) : String;
begin
	if( iFormatting = 0 ) Then
	Begin
		formatCase := s;
	End
	Else
	Begin
		formatCase := LowerCase( s );
	End;
end;

begin
	OutF_Open := False;
	FileCount := 0;
	FSize := 0;

	// Header
	a := mBeg;

	//
	Case iTargetAsm Of

		2 :
		Begin
			Send ( formatCase( 'ORG $' ) + IntToHex ( a, 4 ) + ' - 2' );
			Send ( formatCase( 'DW  $' ) + IntToHex ( a, 4 ) );
		End;

		1 :
		Begin
			Send ( '* = $' + IntToHex ( a, 4 ) );
		End;

		0 :
		Begin
			Send ( #9 + formatCase( 'ORG $' ) + IntToHex ( a, 4 ) + ' - 2' );
			Send ( #9 + formatCase( 'DW  $' ) + IntToHex ( a, 4 ) );
		End;

	End;

	Send ( '; Zeropage vectors' );

	// Zeropage vectors
	Cnt := 0;
	For a := 0 To 255 Do
	Begin
		If ( ZeroPage [ a ] <> 0 ) And ZeroPageVector [ a ] Then
		begin
			s := 'zp' + IntToHex ( a, 2 ) + ' = $' + IntToHex ( a, 2 ) + ' ;(' + IntToStr ( ZeroPage [ a ] ) + ')';
			Send ( s );
			Inc ( Cnt );
		End;
	End;

	If Cnt > 0 Then
		Send ( ';---------' );
	Send ( '; Zeropage variables' );

	// Zeropage variables
	Cnt := 0;
	For a := 0 To 255 Do
	Begin
		If ( ZeroPage [ a ] <> 0 )
		And Not ZeroPageVector [ Abs ( a - 1 ) ]
		And Not ZeroPageVector [ a ] Then
		Begin
			s := 'zp' + IntToHex ( a, 2 ) + ' = $' + IntToHex ( a, 2 ) + ' ;(' + IntToStr ( ZeroPage [ a ] ) + ')';
			Send ( s );
			Inc ( Cnt );
		End;
	End;

	If Cnt > 0 Then
		Send ( ';---------' );

	// Low memory variables
	Cnt := 0;
	For a := $0100 To mBeg - 1 Do
	Begin
		If ( MemData^ [ a ] ) Or ( Labels^ [ a ] <> ltNone ) Then
		Begin
			Send ( destAddr ( a ) + ' = $' + IntToHex ( a, 4 ) );
			Inc ( Cnt );
		End;
	End;

	If Cnt > 0 Then
		Send ( ';---------' );

	// High memory variables
	Cnt := 0;
	For a := mEnd To $FFFF Do
	Begin
		If ( MemData^ [ a ] ) Or ( Labels^ [ a ] <> ltNone ) Then
		Begin
			Send ( destAddr ( a ) + ' = $' + IntToHex ( a, 4 ) );
			Inc ( Cnt );
		End;
	End;

	If Cnt > 0 Then
		Send ( ';---------' );

	// Init
	DataBytes := 0;
	bLineWordMode := False;
	bStringOpen := False;
	bPrevOpcode := False;
	a := mBeg; // start from beginning
	s := '';
	Count := 0;

	// Main loop
	While a < mEnd Do
	Begin
		//
		{
		if a = $8FBF Then begin
			ShowMessage ( 'Debug breakpoint (Address: $' + IntToHex( a, 4 ) + ')' );
		end;
		}
		//
		OpCode := Memory^[a];
		//
		ad := OpCodes [ OpCode ].ad;
		by := OpCodes [ OpCode ].by;
		//
		If ( Labels^ [ a ] <> ltNone )
		And ( ( MemData^ [ a ] )
		Or ( ( by > 1 ) And ( MemData^ [ a + 1 ] ) )
		Or ( ( by > 2 ) And ( MemData^ [ a + 2 ] ) ) ) Then
		Begin
			FlushData;
			//
			s := LabelStr ( 'V_' + IntToHex ( a, 4 ) );
			//
			Case iTargetAsm Of

				2 : Send ( s );

				1 : s := s + ' ';

				0 : Send ( s );

			End;
		End;
		//
		If MemCode^ [ a ] <> mcNone Then
		Begin
			//
			// Deal with illegal opcodes
			//
			If OpCodes [ OpCode ].il = 1 Then
			Begin
				Send ( '; $' + IntToHex ( a, 4 ) );
				MemCode^ [ a ] := mcNone;
				If by > 1 Then
					MemCode^ [ ( a + 1 ) And $FFFF ] := mcNone;
				If by > 2 Then
					MemCode^ [ ( a + 2 ) And $FFFF ] := mcNone;
				//
				If ( a > 3 ) Then
				Begin
					//
					// Check if this illegal instruction is
					// following a JSR $nnnn instruction
					//
					dest := Memory^ [ a - 2 ] Or Word ( Memory^ [ a - 1 ] ) Shl 8;
					If ( dest > 0 ) And ( Memory^ [ a - 3 ] = $20 )
					And ( MemCode^ [ a - 3 ] = mcInstruction ) Then
					Begin
						//
						// Yes, possible Sub with parameter.
						//
						If SubParams [ dest ].SpecialType = ssNone Then
						Begin
							// Increment counter
							Inc ( SubParams [ dest ].Value );
							// ShowMessage ( 'Possible Sub with Params: $' + IntToHex ( dest, 4 ) );
						End;
					End;
				End;
			End;
		End;
		//

		//
		// Deal specifically with BIT $00 and BIT $0000
		//
		If ( ( OpCode = $24 ) And ( Labels^ [ a + 1 ] <> ltNone ) )
		Or ( ( OpCode = $2C ) And ( Labels^ [ a + 1 ] <> ltNone ) ) Then
		Begin
			// turn them into DATA
			MemCode^ [ a ] := mcNone;
			by := 1; // force one byte only
		End;

		If ( ( OpCode = $2C ) And ( Labels^ [ a + 2 ] <> ltNone ) ) Then
		Begin
			// turn them into DATA
			MemCode^ [ a ] := mcNone;
			by := 2; // force two bytes
		End;

		If MemCode^ [ a ] = mcNone Then
		Begin

			// Data
			If DataBytes = 0 Then
			Begin
				s := '';
				GetLabel;
				If ( s = #9 ) Then
				Begin
					s := '';
					// print out address if new data section without a label starts
					if ( bPrevOpcode ) Then
						Send ( ';$' + IntToHex ( a, 4 ) + '  --------' );
				End;
			End;

			// Start new data line if
			// - this address has a label
			// - it's on a 256 byte boundary (and no stopbyte is specified)
			// - it's the beginning of a JumpTable
			//
			If ( MemData^ [ a ] ) Or IsJumpTable ( a )
			Or ( ( ( a And $FF ) = 0 ) And ( DataStyles [ a ].StopByte < 0 ) ) Then
			Begin
				//If ( DataBytes And 15 ) > 0 Then
				If ( DataBytes > 0 ) Then
				Begin
					//
					FlushData;
					//
				End;

				//
				If IsJumpTable ( a ) Then
				Begin
					Send ( '; $' + IntToHex ( a, 4 ) + ' JumpTable' );
				End;

				s := '';

				If Labels^ [ a ] = ltCharset Then
				Begin
					s := LabelStr ( 'C_' + IntToHex ( a, 4 ) );
				End
				Else
				Begin
					s := LabelStr ( 'V_' + IntToHex ( a, 4 ) );
				End;

				Case iTargetAsm Of

					2 :
					Begin
						Send ( s );
						s := '';
					End;

					1 : s := s + ' ';

					0 : // AS65
					Begin
						s := s + #9;
					End;

				End;
				//
				If s <> '' Then
				Begin
					If bAddressNewLine Then
					Begin
						Send ( s );
						s := '';
					End;
				End;
			End;

			//
			If ( IsInsideJumpTable ( a ) > 0 ) And Not MemData^ [ a + 1 ] Then
			Begin
				If ( JumpTables [ IsInsideJumpTable ( a ) ].Style <> 2 )
				And ( JumpTables [ IsInsideJumpTable ( a ) ].Style <> 4 ) Then
				Begin
					bWordMode := True;
					// force opcode size to 2
					by := 2;
				End
				Else
				Begin
					bWordMode := False;
					by := 1;
				End;
			End
			Else
			Begin
				If ( ( a And 1 ) = 0 ) And ( DataStyles [ a ].DataFormat = dfWord )
				And Not MemData^ [ a + 1 ] Then
				Begin
					bWordMode := True;
					// force opcode size to 2
					by := 2;
				End
				Else
				Begin
					bWordMode := False;
					// force opcode size to 1
					by := 1;
				End;
			End;

			If ( DataBytes > 0 ) And ( bLineWordMode <> bWordMode ) Then
			Begin
				FlushData;
			End;
			bLineWordMode := bWordMode;

			// first byte on this line?
			If DataBytes = 0 Then
			Begin
				//
				Case iTargetAsm Of

					2 :
					Begin
						If bWordMode Then
							s := s + formatCase( 'DW ' )
						Else
							s := s + formatCase( 'DB ' );
					End;

					1 :
					Begin
						If bWordMode Then
							s := s + formatCase( '.WORD ' )
						Else
							s := s + formatCase( '.BYTE ' );
					End;

					0 :
					Begin
						If s = '' Then
							s := s + #9;
						If bWordMode Then
							s := s + formatCase( 'DW ' )
						Else
							s := s + formatCase( 'DB ' );
					End;

				End;
			End
			Else
			Begin
				// separate data bytes with comma
				If Not bStringOpen Then
				Begin
					s := s + ','; // except inside strings
				End;
			End;

			//
			Case iTargetAsm Of

				0 :
				Begin
					If bWordMode Then
					Begin
						dest := OpCode + Memory^[a+1] Shl 8;
						If ( IsInsideJumpTable ( a ) > 0 )
						And ( JumpTables [ IsInsideJumpTable ( a ) ].Style = 1 ) Then
						Begin
							s := s + destaddr ( dest + 1 ) + '-1';
						End
						Else
						Begin
							s := s + destaddr ( dest );
						End;
					End
					Else
					Begin
						If ( DataStyles [ a ].DataFormat = dfAuto )
						And ( a > 0 ) And ( a < $FFFF ) And ( IsStringChar ( OpCode ) )
						And ( IsStringChar ( Memory^[a-1] ) Or IsStringChar ( Memory^[a+1] ) ) Then
						Begin
							//
							If bStringOpen Then
							Begin
								s := s + Chr ( OpCode );
							End
							Else
							Begin
								s := s + '"' + Chr ( OpCode );
								bStringOpen := True;
							End;
						End
						Else
						Begin
							//
							CloseString ( ',' );
							//
							If ( DataStyles [ a ].DataFormat = dfAuto )
							And ( a > 0 ) And IsCBMChar ( OpCode )
							And ( IsCBMChar ( Memory^[a-1] ) Or IsCBMChar ( Memory^[a+1] ) ) Then
								// cmb-ascii
								s := s + '"' + Chr ( OpCode + $40 ) + '"-64'
							Else
								s := s + '$' + IntToHex ( OpCode, 2 );
						End;
					End;
				End;

				1, 2 :
				Begin
					If bWordMode Then
					Begin
						s := s + '$' + IntToHex ( OpCode + Memory^[a+1] Shl 8, 4 );
					End
					Else
					Begin
						s := s + '$' + IntToHex ( OpCode, 2 );
					End;
				End;

			End;
			//
			Inc ( DataBytes, by );

			//
			// -- line "full"?
			//
			If ( ( DataStyles [ a ].StopByte < 0 ) And ( ( DataBytes Mod DataStyles [ a ].Columns ) = 0 ) )
			Or ( ( DataStyles [ a ].StopByte >= 0 ) And ( Memory^ [ a ] = DataStyles [ a ].StopByte ) )
			Or ( MemCode^ [ ( a + by ) And $FFFF ] <> mcNone ) Then
			Begin
				//
				FlushData;
				//Send ( '; $' + IntToHex ( a + by, 4 ) );
				//
			End;

			bPrevOpcode := false; // flag that this was data

		End
		Else
		Begin

			s := '';
			sComm := '';

			//
			// Code, not data
			//

			If False Then
			Begin

			(*
				// unused code, MONITOR style output
				s := '>' + IntToHex ( a, 4 ) + ' ';
				For b := 1 To by Do
				Begin
					s := s + IntToHex ( Memory^[a+b-1], 2 ) + ' ';
				End;
				For b := by + 1 To 3 Do
				Begin
					s := s + '   ';
				End;
				*)

			End
			Else
			Begin

				GetLabel;

			End;

			if ( Not bPrevOpcode ) Then
			Begin
				Send ( #9 + ';' );
			End;

			If ( s <> '' ) And ( s <> #9 ) Then
			Begin
				If bAddressNewLine Then
				Begin
					Send ( s );
					s := #9;
				End;
			End;

			If iFormatting = 0 Then
				s := s + UpperCase ( OpCodes [ OpCode ].mn )
			Else
				s := s + LowerCase ( OpCodes [ OpCode ].mn );

			//
			{
			If OpCode = 0 Then
			Begin
				While Memory^[a+1] = 0 Do
					Inc ( a );
			End;
			}
			//
			Case ad Of

				0 : ; { none }

				1 : { 16 bit direct address }
				Begin
					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;
					s := s + ' ' + destAddr ( dest );
					destComment ( dest, s );
				End;

				2 : { 8 bit direct value }
				Begin
					value := Memory^[a+1];
					//
					If ( a < $FFFF - 4 ) And ( Memory^ [ a + 2 ] = $8D ) Then
					Begin
						dest := Memory^[a+3] Or Word(Memory^[a+4])Shl 8;

						Case dest Of

							$0312 :
							Begin
								dest := FindVec ( value, False );
								If dest = 0 Then
									s := s + ' #$' + IntToHex ( value, 2 )
								Else
									s := s + ' #' + LoByteStr ( 'I_' + IntToHex ( dest, 4 ) );
							End;

							$0313 :
							Begin
								dest := FindVec ( value, True );
								If dest = 0 Then
									s := s + ' #$' + IntToHex ( value, 2 )
								Else
									s := s + ' #' + HiByteStr ( 'I_' + IntToHex ( dest, 4 ) );
							End;

							$0314 :
							Begin
								dest := FindIRQ ( value, False );
								If dest = 0 Then
									s := s + ' #$' + IntToHex ( value, 2 )
								Else
									s := s + ' #' + LoByteStr ( 'I_' + IntToHex ( dest, 4 ) );
							End;

							$0315 :
							Begin
								dest := FindIRQ ( value, True );
								If dest = 0 Then
									s := s + ' #$' + IntToHex ( value, 2 )
								Else
									s := s + ' #' + HiByteStr ( 'I_' + IntToHex ( dest, 4 ) );
							End;

							$FF13 :
							Begin
								If Labels^ [ value Shl 8 ] = ltCharset Then
								Begin
									s := s + ' #' + HiByteStr ( 'C_' + IntToHex ( value, 2 ) + '00' )
								End
								Else
								Begin
									s := s + ' #$' + IntToHex ( value, 2 );
								End;
								//
								sComm := 'Set charset';
							End;

							$FFFE :
							Begin
								dest := FindNMI ( value, False );
								If dest = 0 Then
									s := s + ' #$' + IntToHex ( value, 2 )
								Else
									s := s + ' #' + LoByteStr ( 'N_' + IntToHex ( dest, 4 ) );
							End;

							$FFFF :
							Begin
								dest := FindNMI ( value, True );
								If dest = 0 Then
									s := s + ' #$' + IntToHex ( value, 2 )
								Else
									s := s + ' #' + HiByteStr ( 'N_' + IntToHex ( dest, 4 ) );
							End;

							Else
							Begin
								s := s + ' #$' + IntToHex ( value, 2 );
							End;
						End;

					End
					Else
					Begin
						//
						s := s + ' #$' + IntToHex ( value, 2 );
						//
					End;
					//
					// *** Detect Word Values
					//
					If ( a + 8 < $FFFF ) Then // for zeropage addresses
					Begin
						// LDA #$00 STA $00 pairs for A, X, Y
						If ( ( OpCode = $A9 ) And ( Memory^ [ a + 2 ] = $85 ) )
						Or ( ( OpCode = $A2 ) And ( Memory^ [ a + 2 ] = $86 ) )
						Or ( ( OpCode = $A0 ) And ( Memory^ [ a + 2 ] = $84 ) ) Then
						Begin
							// LDA #$00 STA $00 pairs for A, X, Y (2nd)
							If ( ( Memory^ [ a + 4 ] = $A9 ) And ( Memory^ [ a + 6 ] = $85 ) )
							Or ( ( Memory^ [ a + 4 ] = $A2 ) And ( Memory^ [ a + 6 ] = $86 ) )
							Or ( ( Memory^ [ a + 4 ] = $A0 ) And ( Memory^ [ a + 6 ] = $84 ) ) Then
							Begin
								// low byte, high byte?
								If ( Memory^ [ a + 3 ] + 1 = Memory^ [ a + 7 ] ) Then
								Begin
									dest := Memory^ [ a + 5 ] Shl 8 + Memory^ [ a + 1 ];
									MemData^ [ dest ] := True;
									sComm := sComm + '$' + IntToHex ( dest, 4 ) + ' to $' + IntToHex ( Memory^ [ a + 3 ], 2 ) + ' (16bit value)';
								End;

								// high byte, low byte?
								If ( Memory^ [ a + 3 ] = Memory^ [ a + 7 ] + 1 ) Then
								Begin
									dest := Memory^ [ a + 1 ] Shl 8 + Memory^ [ a + 5 ];
									MemData^ [ dest ] := True;
									sComm := sComm + '$' + IntToHex ( dest, 4 ) + ' to $' + IntToHex ( Memory^ [ a + 7 ], 2 ) + ' (16bit value)';
								End;
							End;
						End;
					End;
					If ( a + 10 < $FFFF ) Then // for 16 bit addresses
					Begin
						// LDA #$00 STA $00 pairs for A, X, Y
						If ( ( OpCode = $A9 ) And ( Memory^ [ a + 2 ] = $8D ) )
						Or ( ( OpCode = $A2 ) And ( Memory^ [ a + 2 ] = $8E ) )
						Or ( ( OpCode = $A0 ) And ( Memory^ [ a + 2 ] = $8C ) ) Then
						Begin
							// LDA #$00 STA $00 pairs for A, X, Y (2nd)
							If ( ( Memory^ [ a + 5 ] = $A9 ) And ( Memory^ [ a + 7 ] = $8D ) )
							Or ( ( Memory^ [ a + 5 ] = $A2 ) And ( Memory^ [ a + 7 ] = $8E ) )
							Or ( ( Memory^ [ a + 5 ] = $A0 ) And ( Memory^ [ a + 7 ] = $8C ) ) Then
							Begin
								// low byte, high byte?
								If ( Memory^ [ a + 3 ] + 1 = Memory^ [ a + 8 ] )
								And ( Memory^ [ a + 4 ] = Memory^ [ a + 9 ] ) Then
								Begin
									dest := Memory^ [ a + 6 ] Shl 8 + Memory^ [ a + 1 ];
									MemData^ [ dest ] := True;
									sComm := sComm + '$' + IntToHex ( dest, 4 ) + ' to $' + IntToHex ( Memory^ [ a + 4 ], 2 ) + IntToHex ( Memory^ [ a + 3 ], 2 ) + ' (16bit value)';
								End;

								// high byte, low byte?
								If ( Memory^ [ a + 3 ] = Memory^ [ a + 8 ] + 1 )
								And ( Memory^ [ a + 4 ] = Memory^ [ a + 9 ] ) Then
								Begin
									dest := Memory^ [ a + 1 ] Shl 8 + Memory^ [ a + 6 ];
									MemData^ [ dest ] := True;
									sComm := sComm + '$' + IntToHex ( dest, 4 ) + ' to $' + IntToHex ( Memory^ [ a + 9 ], 2 ) + IntToHex ( Memory^ [ a + 8 ], 2 ) + ' (16bit value)';
								End;
							End;
						End;
					End;
					//
					// *** Detect Keyboard Query ***
					//
					If ( OpCode = $A9 ) And ( a < $FFFF - 5 ) Then
					Begin
						// check for LDA #$nn, STA $FDxx
						If ( Memory^ [ a + 2 ] = $8D ) And ( Memory^ [ a + 4 ] = $FD ) Then
						Begin
							sComm := sComm + 'Check column ';
							For Bits := 0 To 7 Do
							Begin
								If ( value And ( $01 Shl Bits ) ) = 0 Then
									sComm := sComm + IntToStr ( Bits ) + ', ';
							End;
							sComm := RemoveFromRight ( sComm, 2 );
						End;
					End;
					//
					If ( OpCode = $29 ) And ( a > 11 ) Then
					Begin
						// check for LDA #$nn, STA $FDxx, STA $FF08, LDA $FF08, AND #$nn
						If ( Memory^[a-11] = $A9 )
						And ( Memory^[a-9] = $8D ) And ( Memory^[a-7] = $FD )
						And ( Memory^[a-6] = $8D ) And ( Memory^[a-5] = $08 ) And ( Memory^[a-4] = $FF )
						And ( Memory^[a-3] = $AD ) And ( Memory^[a-2] = $08 ) And ( Memory^[a-1] = $FF ) Then
						Begin
							KeyColumn := -1;
							For Bits := 0 To 7 Do
							Begin
								If ( Memory ^ [ a - 10 ] = ( $FF XOR ( $01 Shl Bits ) ) ) Then
								Begin
									KeyColumn := Bits;
								End;
							End;
							If KeyColumn >= 0 Then
							Begin
								//
								sComm := sComm + 'Query keyboard for ';
								For Bits := 0 To 7 Do
								Begin
									If ( value And ( $01 Shl Bits ) ) <> 0 Then
									Begin
										sComm := sComm + '"' + KeyboardMatrix [ Bits * 8 + KeyColumn ] + '", ';
									End;
								End;
								sComm := RemoveFromRight ( sComm, 2 );
								//
							End;
						End;
					End;
					//
					// *** Detect Joystick Query ***
					//
					If ( OpCode = $A9 ) And ( a + 8 < $FFFF ) Then
					Begin
						// check for LDA #$nn, STA $FF08, LDA $FF08
						If ( Memory^[a+2] = $8D ) And ( Memory^[a+3] = $08 ) And ( Memory^[a+4] = $FF )
						And ( Memory^[a+5] = $AD ) And ( Memory^[a+6] = $08 ) And ( Memory^[a+7] = $FF ) Then
						Begin
							sComm := sComm + 'Query joystick ';
							//
							Case ( value And $03 ) Of
								$00 : sComm := sComm + '#1 and #2';
								$01 : sComm := sComm + '#2';
								$02 : sComm := sComm + '#1';
							End;
						End;
					End;
					//
					// *** Check for color codes
					//
					If ( a + 5 < $FFFF ) Then
					Begin
						If ( Memory^ [ a + 4 ] = $FF )
						And ( Memory^ [ a + 3 ] >= $15 )
						And ( Memory^ [ a + 3 ] <= $19 ) Then
						Begin
							If ( ( OpCode = $A9 ) And ( Memory^ [ a + 2 ] = $8D ) )
							Or ( ( OpCode = $A2 ) And ( Memory^ [ a + 2 ] = $8E ) )
							Or ( ( OpCode = $A0 ) And ( Memory^ [ a + 2 ] = $8C ) ) Then
							Begin
								sComm := sComm + 'Color: ' + ColorCodes [ value And $0F ];
								If ( value And $0F ) <> 0 Then
								Begin
									// add luminance for non-black
									sComm := sComm + ' luminance ' + IntToStr ( ( value Shr 4 ) And 15 );
								End;
							End;
						End;
					End;
					//
					// *** Detect vertical sync
					//
					If ( a > 2 ) And ( Memory^ [ a - 2 ] = $1D ) And ( Memory^ [ a - 1 ] = $FF ) Then
					Begin
						If ( ( OpCode = $C9 ) And ( Memory^ [ a - 3 ] = $AD ) )
						Or ( ( OpCode = $E0 ) And ( Memory^ [ a - 3 ] = $AE ) )
						Or ( ( OpCode = $C0 ) And ( Memory^ [ a - 3 ] = $AC ) ) Then
						Begin
							If ( ( Memory^ [ a + 2 ] = $D0 ) And ( Memory^ [ a + 3 ] = $F9 ) )
							Or ( ( Memory^ [ a + 2 ] = $90 ) And ( Memory^ [ a + 3 ] = $F9 ) ) Then
							Begin
								sComm := sComm + 'Sync to vertical rasterline $' + IntToHex ( value, 2 );
							End
							Else
							Begin
								sComm := sComm + 'Vertical rasterline $' + IntToHex ( value, 2 );
							End;
						End;
					End;
					//
					// *** Generic, ascii codes... ***
					//
					if ( value >= $41 ) And ( value <= $59 ) Then
					Begin
						sComm := sComm + '"' + Chr ( value ) + '"';
					End;
					//
				End;

				3 : { 16 bit address + X }
				Begin
					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;
					s := s + ' ' + destAddr ( dest ) + formatCase( ',X' );
					destComment ( dest, s );
				End;

				4 : { relative (8 bit) address }
				Begin
					rel := ShortInt ( Memory^[a+1] );
					dest := a + 2 + rel;
					s := s + ' ' + destAddr ( dest );
				End;

				5 : { zero page address + X }
				Begin
					If ( a < $FFFF ) Then
					Begin
						dest := Memory^[a+1];
						s := s + ' ' + ZeroPageAddress ( dest ) + formatCase( ',X' );
					End;
				End;

				6 : { zero page address }
				Begin
					dest := Memory^[a+1];
					s := s + ' ' + ZeroPageAddress ( dest );
				End;

				7 : { 16 bit address + Y }
				Begin
					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;
					s := s + ' ' + destAddr ( dest ) + formatCase( ',Y' );
					destComment ( dest, s );
				End;

				8 : { indirect 16 bit address (JMP) }
				Begin
					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;
					s := s + ' (' + destAddr ( dest ) + ')';
					destComment ( dest, s );
					//
					sComm := '*** Indirect jump!';
				End;

				9 : { indirect zero page address + Y }
				Begin
					dest := Memory^[a+1];
					s := s + ' (' + ZeroPageAddress ( dest ) + formatCase( '),Y' );
				End;

				10 : { indirect zero page address + X }
				Begin
					dest := Memory^[a+1];
					s := s + ' (' + ZeroPageAddress ( dest ) + formatCase( ',X)' );
				End;

				12 : { zero page address + Y }
				Begin
					dest := Memory^[a+1];
					s := s + ' ' + ZeroPageAddress ( dest ) + formatCase( ',Y' );
				End;

				15 : { "A" }
				Begin
					s := s + formatCase( ' A' );
				End;

			End;
			//
			For iComment := 1 To nComments Do
			Begin
				iCommentBytes := 1;
				bCommentMatch := True;
				While bCommentMatch And ( iCommentBytes <= Comments [ iComment ].nCommentBytes ) Do
				Begin
					bCommentPos := Comments [ iComment ].CommentBytes [ iCommentBytes ].Position;
					// make sure the checked bytes are inside the memory
					If ( a + bCommentPos >= 0 ) And ( a + bCommentPos <= $FFFF ) Then
					Begin
						bCommentByte := Memory^ [ a + bCommentPos ];
						bCommentMatch := MatchCommentByte ( bCommentByte,
							Comments [ iComment ].CommentBytes [ iCommentBytes ].MatchValue );
						Inc ( iCommentBytes );
					End
					Else
					Begin
						bCommentMatch := False;
					End;
				End;
				//
				If bCommentMatch Then
				Begin
					// it's a match, add it!
					if sComm <> '' Then
						sComm := sComm + ' - ';
					sComm := sComm + ProcessComment ( Comments [ iComment ].Comment, a );
				End;
			End;
			//
			If sComm <> '' Then
			Begin
				If Pos ( ';', s ) = 0 Then
					s := s + #9 + '; '
				Else
					s := s + ' - ';
				s := s + sComm;
			End;
			//
			Send ( s );
			s := '';
			//

			// RTS or RTI
			If ( OpCode = $60 ) Or ( OpCode = $40 ) Then
			Begin
				Send ( ';$' + IntToHex ( a + 1, 4 ) + '  --------' );
			End
			Else
			Begin
				If ( OpCode = $4C ) Or ( OpCode = $6C ) Then
				Begin
					Send ( #9 + ';' ); // separate jumps
				End;
			End;

			bPrevOpcode := true; // flag that this was an opcode
		End;

		//
		a := a + by;
		//
		Inc ( Count );
		If ( Count And $3 ) = 0 Then
		Begin
			Application.ProcessMessages;
		End;
	End;

	If ( DataBytes > 0 ) And ( s <> '' ) Then
	Begin
		FlushData;
	End;

	Send ( '; EOF' );

	If OutF_Open Then
	Begin
		CloseFile ( OutF );
	End;

	// Check for possible sub params
	s := '';
	For a := 0 To $FFFF Do
	Begin
		If ( SubParams [ a ].SpecialType = ssNone )
		And ( SubParams [ a ].Value > 0 ) Then
		Begin
			s := s + '$' + IntToHex ( a, 4 ) + ' (' + IntToStr ( SubParams [ a ].Value ) + ' times)' + #13 + #10;
			sPSubParams := sPSubParams + ';$' + IntToHex ( a, 4 ) + ', Length, 2' + #13 + #10;
		End;
	End;
	If s <> '' Then
	Begin
		ShowMessage ( 'Possible Subs with Params:' + #13 + #10 + s );
	End;
end;

// Analyze the code
procedure TFormMain.RunCode ( StartAddr : Word; lt : TLabelType );
var
	a, Count, i : Integer;
	OpCode, by, b : Byte;
	rel : Shortint; // for relative addresses
	dest : Word;
	Stopped : Boolean;
	//
	LoIRQ, HiIRQ : Byte;
	LoIRQValid, HiIRQValid : Boolean;
	//
	LoVec, HiVec : Byte;
	LoVecValid, HiVecValid : Boolean;
	//
	LoRes, HiRes : Byte;
	LoResValid, HiResValid : Boolean;
	//
	LoNMI, HiNMI : Byte;
	LoNMIValid, HiNMIValid : Boolean;
	//
	aReg, xReg, yReg, RegValue : Byte;
begin
	//
	LoVec := 0;
	HiVec := 0;
	LoVecValid := False;
	HiVecValid := False;
	//
	LoIRQ := 0;
	HiIRQ := 0;
	LoIRQValid := False;
	HiIRQValid := False;
	//
	LoRes := 0;
	HiRes := 0;
	LoResValid := False;
	HiResValid := False;
	//
	LoNMI := 0;
	HiNMI := 0;
	LoNMIValid := False;
	HiNMIValid := False;
	//
	aReg := 0;
	xReg := 0;
	yReg := 0;
	a := StartAddr;
	Stopped := False;
	//
	Labels^ [ a ] := lt; // mark start address
	//
	Count := 0;
	While Not Stopped Do
	Begin
		If ( MemCode^ [ a ] <> mcNone ) Or ( HardStops^ [ a ] ) Then
		Begin
			// reached code that was already processed, so exit
			// or hard stop was specified by the user.
			Stopped := True;
		End
		Else
		Begin

			if a = $1b9b then
			begin
				aReg := xReg; // debug breakpoint
			end;
			MemoPreview.Lines.Add( '$' + IntToHex( a, 4 ) );

			OpCode := Memory^[a];

			by := OpCodes [ OpCode ].by;

			{ mark first by as Instruction }
			MemCode^ [ a ] := mcInstruction;

			{ mark following bytes (if any) as params }
			if ( a < $FFFF - by ) then
			Begin
				For b := 1 To by - 1 Do
					MemCode^ [ a + b ] := mcParams;
			End;

			Case OpCode Of

				$20 : { JSR }
				Begin
					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;

					If ( dest < $0200 ) Then
					Begin
						// ShowMessage ( 'JSR to $' + IntToHex ( dest, 4 ) );
					End
					Else
					Begin
						RunCode ( dest, ltSub );
						//
						If SubParams [ dest ].SpecialType <> ssNone Then
						Begin
							Case SubParams [ dest ].SpecialType Of
								ssLength :
								Begin
									Inc ( a, SubParams [ dest ].Value );
									//
									// hack! 2 byte param assumed data address...
									//
									If SubParams [ dest ].Value = 2 Then
									begin
										dest := Memory^[a+1] Or Word(Memory^[a+2]) Shl 8;
										MemData^ [ dest ] := True;
									End;
									//
								End;
								ssStopByte :
								Begin
									While Memory^[ a + by ] <> SubParams [ dest ].Value Do
									Begin
										Inc ( a );
									End;
									Inc ( a );
								End;
							End;
						End;
					End;
				End;

				$4C : { JMP }
				Begin
					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;

					If ( dest < $0200 ) Or ( dest > mEnd ) Then
					Begin
						// ShowMessage ( 'JMP to $' + IntToHex ( dest, 4 ) );
						MemData^ [ dest ] := True;
					End
					Else
					Begin
						RunCode ( dest, ltJump );
					End;

					Stopped := True;
				End;

				$00, $40, $60, $6C : { BRK, RTI, RTS, JMP () }
				Begin
					Stopped := True;
				End;

				$02, $12, $22, $32,
				$42, $52, $62, $72,
				$92, $B2, $D2, $F2 : { JAM }
				Begin
					Stopped := True;
				End;

				$84, $85, $86 : { STY, STA, STX $nn }
				Begin
					// check for zeropage vectors?
				End;

				$8C, $8D, $8E : { STY, STA, STX $nnnn }
				Begin

					Case OpCode Of
						$8C : RegValue := yReg;
						$8D : RegValue := aReg;
						Else  RegValue := xReg;
					End;

					dest := Memory^[a+1] Or Word(Memory^[a+2])Shl 8;
					If ( dest < $FD00 ) Then
						MemData^ [ dest ] := True;

					Case dest Of

						$0312:
						Begin
							LoVec := RegValue;
							LoVecValid := True;
						End;

						$0313:
						Begin
							HiVec := RegValue;
							HiVecValid := True;
						End;

						$0314:
						Begin
							LoIRQ := RegValue;
							LoIRQValid := True;
						End;

						$0315:
						Begin
							HiIRQ := RegValue;
							HiIRQValid := True;
						End;

						$FFFC:
						Begin
							LoRes := RegValue;
							LoResValid := True;
						End;

						$FFFD:
						Begin
							HiRes := RegValue;
							HiResValid := True;
						End;

						$FFFE:
						Begin
							LoNMI := RegValue;
							LoNMIValid := True;
						End;

						$FFFF:
						Begin
							HiNMI := RegValue;
							HiNMIValid := True;
						End;

						$FF13 :
						Begin
							{ charset address : $nn00 }
							dest := Word ( RegValue And $FC ) Shl 8;
							Labels^ [ dest ] := ltCharset;
							MemData^ [ dest ] := True;
							//
							For i := dest To dest + $07FF Do
							Begin
								DataStyles[ i ].DataFormat := dfByte;
								DataStyles[ i ].Columns := 8;
								DataStyles[ i ].StopByte := -1;
							End;
						End;

					End;

					If LoVecValid And HiVecValid Then
					Begin
						dest := LoVec Or Word ( HiVec ) Shl 8;
						RunCode ( dest, ltVec );
						LoVecValid := False;
						HiVecValid := False;
					End;

					If LoIRQValid And HiIRQValid Then
					Begin
						dest := LoIRQ Or Word ( HiIRQ ) Shl 8;
						RunCode ( dest, ltIRQ );
						LoIRQValid := False;
						HiIRQValid := False;
					End;

					If LoResValid And HiResValid Then
					Begin
						dest := LoRes Or Word ( HiRes ) Shl 8;
						RunCode ( dest, ltRes );
						LoResValid := False;
						HiResValid := False;
					End;

					If LoNMIValid And HiNMIValid Then
					Begin
						dest := LoNMI Or Word ( HiNMI ) Shl 8;
						RunCode ( dest, ltNMI );
						LoNMIValid := False;
						HiNMIValid := False;
					End;

				End;

				$A0 : { LDY #$nn }
				Begin
					yReg := Memory^[a+1];
					if ( Memory^[a+2] = $84 ) Then
					Begin
					End;
				End;

				$A2 : { LDX #$nn }
				Begin
					xReg := Memory^[a+1];
					if ( Memory^[a+2] = $86 ) Then
					Begin
					End;
				End;

				$A9 : { LDA #$nn }
				Begin
					aReg := Memory^[a+1];
					if ( Memory^[a+2] = $85 ) Then
					Begin
					End;
				End;

			End;

			// now check by addressing mode

			Case OpCodes [ OpCode ].ad Of

				1, 3, 7 : { 16 mem operation }
				Begin
					dest := Memory^[ ( a + 1 ) And $FFFF ] Or Word(Memory^[ ( a + 2 ) And $FFFF ])Shl 8;
					If dest < $FD00 Then
					Begin
						//
						// except jumps!
						If ( OpCode <> $20 )
						And ( OpCode <> $4C )
						And ( OpCode <> $6C )
						Then
						Begin
							//
							If ( dest > 0 ) And IsJumpTable ( dest - 1 ) Then
								Dec ( dest );
							MemData^ [ dest ] := True;
							//
						End;
					End;
				End;

				5, 6, 9, 10, 12 : { $nn,x $nn ($nn),y ($nn,x) $nn,y }
				Begin
					If ( a < $FFFF ) Then
					Begin
						dest := Memory^[a+1];
						Inc ( ZeroPage [ dest ] );
						//
						// check for indirect indexing mode
						If OpCodes [ OpCode ].ad = 9 Then
						Begin
							ZeroPageVector [ dest ] := True;
						End;
					End;
				End;

				4: { branch? }
				Begin
					rel := ShortInt ( Memory^[a+1] );
					dest := a + 2 + rel;
					RunCode ( dest, ltBranch );
					//
					// BPL/BMI
					// BEQ/BNE
					// BCC/BCS
					//
					If ( ( OpCode = $10 ) And ( Memory^ [ a - 2 ] = $30 ) )
					Or ( ( OpCode = $30 ) And ( Memory^ [ a - 2 ] = $10 ) )
					Or ( ( OpCode = $D0 ) And ( Memory^ [ a - 2 ] = $F0 ) )
					Or ( ( OpCode = $F0 ) And ( Memory^ [ a - 2 ] = $D0 ) )
					Or ( ( OpCode = $90 ) And ( Memory^ [ a - 2 ] = $B0 ) )
					Or ( ( OpCode = $B0 ) And ( Memory^ [ a - 2 ] = $90 ) ) Then
					Begin
						// make sure the previous instruction is a 2 byte short branch
						If Labels^ [ dest - 2 ] = ltBranch Then
						Begin
							Stopped := True;
						End;
					End;
					//
					// SEC/BCS
					// CLC/BCC
					//
					If ( ( OpCode = $B0 ) And ( Memory^ [ a - 1 ] = $38 ) )
					Or ( ( OpCode = $90 ) And ( Memory^ [ a - 1 ] = $18 ) ) Then
					Begin
						// make sure the previous instruction is a 1 byte instruction (not data)
						If MemCode^ [ a - 1 ] = mcInstruction Then
						Begin
							Stopped := True;
						End;
					End;
				End;

			End;

			a := a + by;
			If ( a >= mEnd ) Then
				Stopped := True;
			//
			Application.ProcessMessages; // give time to animation
		End;
	End;
end;

procedure TFormMain.DumpCode;
var
	a, Branches : Integer;
	s : String;
begin
	MemoStats.Lines.Clear;
	//
	a := mBeg;
	While a < mEnd Do
	Begin
		While ( a < mEnd ) And ( MemCode^[a] = mcNone ) Do
			Inc ( a );

		If a < mEnd Then
		Begin
			s := '$' + IntToHex ( a, 4 );
			While ( a < mEnd ) And ( MemCode^[a] <> mcNone ) Do
			Begin
				Inc ( a );
			End;
			s := s + ' - $' + IntToHex ( a - 1, 4 );
			MemoStats.Lines.Add ( s );
		End;
	End;

	Branches := 0;
	a := mBeg;
	While a < mEnd Do
	Begin
		s := '$' + IntToHex ( a, 4 ) + ' ';

		Case Labels^[a] Of
			ltNone   : s := '';
			ltJump   : s := s + 'Jump';
			ltSub    : s := s + 'Sub ---';
			ltVec    : s := s + 'Vec ****************';
			ltBranch : Begin s := ''; Inc ( Branches ); End;
		End;

		If s <> '' Then
			MemoStats.Lines.Add ( s );
		Inc ( a );
		Application.ProcessMessages;
	End;
	MemoStats.Lines.Add ( 'Plus ' + IntToStr ( Branches ) + ' branches.' );
end;

procedure TFormMain.EditDestFolderExit(Sender: TObject);
begin
	If DirectoryExists ( EditDestFolder.Text ) Then
	Begin
		sDestDir := EditDestFolder.Text;
		BrowseFolder ( sDestDir );
	End
	Else
	Begin
		EditDestFolder.Text := sDestDir;
	End;
end;

procedure TFormMain.EditTargetExtExit(Sender: TObject);
begin
	sTargetExt := EditTargetExt.Text;
end;

procedure TFormMain.EditSizeLimitChange(Sender: TObject);
begin
	iBreakSize := UpDownSizeLimit.Position;
end;

procedure TFormMain.UpDownSizeLimitClick(Sender: TObject;
	Button: TUDBtnType);
begin
	iBreakSize := UpDownSizeLimit.Position;
end;

procedure TFormMain.cmdBrowseClick(Sender: TObject);
begin
	mnuFileOpenClick ( Sender );
end;

procedure TFormMain.cmdStartClick(Sender: TObject);
begin
	DisAssemble;
end;

Procedure TFormMain.DisAssemble;
Var
	s, sAVI : String;
	bError : Boolean;
begin
	//
	// ---
	//
	s := EditSource.Text;
	If s = '' Then
	Begin
		MessageDlg ( 'Please select a file.', mtInformation, [mbOK], 0 );
		EditSource.SetFocus;
	End
	Else
	Begin
		If Pos ( '\', s ) = 0 Then
			s := EditDestFolder.Text + s;
		//
		// Hide page control, show working animation
		sAVI := Application.ExeName;
		sAVI := Copy ( sAVI, 1, PosR ( '\', sAVI ) ) + 'WORKING.AVI';
		If FileExists ( sAVI ) Then
		Begin
			Animate1.FileName := sAVI;
			Animate1.Active := True;
		End;
		Animate1.Visible := True;
		//PageControlMain.Visible := False;
		//
		bError := DisAssembleFile ( s );
		//
		// ---
		//
		Animate1.Stop;
		//
		PageControlMain.Visible := True;
		If Not bError Then
			PageControlMain.ActivePageIndex := 1;
	End;
End;

procedure TFormMain.mnuRegClick(Sender: TObject);

procedure RegType ( sExt : String );
Var
	r : TRegistry;
	sType : String;
	sCommand : String;
Begin
	r := TRegistry.Create;
	r.RootKey := HKEY_CLASSES_ROOT;
	//
	If r.OpenKey ( '.' + sExt, False ) Then
	Begin

		// Already exists
		sType := r.ReadString ( '' );

	End
	Else
	Begin

		// Create it now
		sType := sExt + 'File';

		r.CreateKey ( '.' + sExt );
		r.OpenKey ( '.' + sExt, True );
		r.WriteString ( '', sType );

	End;
	r.CloseKey;
	//
	If Not r.OpenKey ( sType, False ) Then
	Begin
		r.CreateKey ( sType );
		r.OpenKey ( sType, False );
		r.WriteString ( '', sExt + ' File' );
	End;
	r.CloseKey;
	//
	// --- Create default icon if file type doesn't have one yet.
	//
	If Not r.OpenKey ( sType + '\DefaultIcon', False ) Then
	Begin
		r.CreateKey ( sType + '\DefaultIcon' );
		r.OpenKey ( sType + '\DefaultIcon', False );
		r.WriteString ( '', ParamStr ( 0 ) + ',1' );
	End;
	r.CloseKey;
	//
	If Not r.OpenKey ( sType + '\shell', False ) Then
	Begin
		r.CreateKey ( sType + '\shell' );
	End;
	r.CloseKey;
	//
	sCommand := 'DisAss';
	//
	If Not r.OpenKey ( sType + '\shell\' + sCommand, False ) Then
	Begin
		r.CreateKey ( sType + '\shell\' + sCommand );
		r.OpenKey ( sType + '\shell\' + sCommand, False );
		r.WriteString ( '', 'DisAssemble' );
		r.CloseKey;
		//
		r.CreateKey ( sType + '\shell\' + sCommand + '\command' );
		r.OpenKey ( sType + '\shell\' + sCommand + '\command', False );
		r.WriteString ( '', ParamStr ( 0 ) + ' "%1"' );
	End
	Else
	Begin
		r.CloseKey;
		r.OpenKey ( sType + '\shell\' + sCommand + '\command', True );
		r.WriteString ( '', ParamStr ( 0 ) + ' "%1"' );
	End;
	r.CloseKey;
	//
	r.Free;
end;

Begin
	RegType ( 'PRG' );
End;

procedure TFormMain.FormActivate(Sender: TObject);
Var
	s : String;
begin
	If ParamCount > 0 Then
	Begin
		s := ParamStr ( 1 );
		//
		If UpperCase ( KeepFromRight ( s, 7 ) ) = '.DISASS' Then
			s := Copy ( s, 1, Length ( s ) - 7 );  
		//
		EditSource.Text := s;
		sDestDir := Copy ( s, 1, PosR ( '\', s ) );
		EditDestFolder.Text := sDestDir;
	End;
	//
	If EditSource.Text = '' Then
	Begin
		cmdBrowse.SetFocus;
	End
	Else
	Begin
		cmdStart.SetFocus;
	End;
End;

procedure TFormMain.optTargetClick(Sender: TObject);
begin
	iTargetAsm := TRadioButton ( Sender ).Tag;
end;

procedure TFormMain.EditSourceClick(Sender: TObject);
begin
	If EditSource.ItemIndex = -1 Then
	Begin
		EditSource.ItemIndex := 0;
	End;
end;

procedure TFormMain.EditDestFolderKeyPress(Sender: TObject; var Key: Char);
begin
	If Key = #13 Then
	Begin
		Key := #0;
		If DirectoryExists ( EditDestFolder.Text ) Then
		Begin
			BrowseFolder ( EditDestFolder.Text );
		End;
	End;
end;

procedure TFormMain.EditSourceKeyPress(Sender: TObject; var Key: Char);
begin
	If Key = #13 Then
	Begin
		If KeepFromRight ( EditSource.Text, 1 ) = '\' Then
		begin
			sDestDir := EditSource.Text;
			EditDestFolder.Text := sDestDir;
			EditDestFolderKeyPress ( Sender, Key );
			EditSource.ItemIndex := 0;
			Key := #0;
		End
		Else
		Begin
			Key := #0;
			cmdStart.SetFocus;
			cmdStartClick ( Sender );
		End;
	End;
end;

procedure TFormMain.FormKeyPress(Sender: TObject; var Key: Char);
begin
	If Key = #27 Then
	Begin
		Key := #0;
		Close;
	End;
end;

procedure TFormMain.FormWindowProc ( var Message : TMessage );
begin
	if Message.Msg = WM_DROPFILES then
		HandleDroppedFiles ( Message ); // handle WM_DROPFILES message
	DefaultWindowProc ( Message );
end;

procedure TFormMain.HandleDroppedFiles ( var Msg : TMessage );
var
	pcFileName: PChar;
	i, iSize, iFileCount: integer;
begin
	pcFileName := ''; // to avoid compiler warning message
	iFileCount := DragQueryFile( Msg.wParam, $FFFFFFFF, pcFileName, 255 );
	//
	i := 0;
	While i < iFileCount Do
	Begin
		iSize := DragQueryFile(Msg.wParam, i, nil, 0) + 1;
		pcFileName := StrAlloc ( iSize );
		DragQueryFile(Msg.wParam, i, pcFileName, iSize);
		//
		if FileExists(pcFileName) then
		Begin
			If ( LowerCase ( KeepFromRight ( pcFileName, 4 ) ) = '.prg' )
			Or ( i = iFileCount - 1 ) Then
			Begin
				// handle the file
				BrowseFile ( pcFileName );
				// exit from loop, as we can only do one file at a time
				i := iFileCount;
			End
		End;
		StrDispose(pcFileName);
	End;
	DragFinish(Msg.wParam);
end;

procedure TFormMain.optFormattingClick(Sender: TObject);
begin
	iFormatting := TRadioButton( Sender ).Tag;
end;

procedure TFormMain.cbAddressNewLineClick(Sender: TObject);
begin
	bAddressNewLine := cbAddressNewLine.Checked; 
end;

procedure TFormMain.cbExtraCommentsClick(Sender: TObject);
begin
	bExtraComments := cbExtraComments.Checked;
end;

end.
